#include <stdio.h>
#include <math.h>
#include "matrix.h"

double cos_theta, sin_theta;
double theta_radians;

/*-----------------------------------------------------*/
void MultiplyMatrix(Matrix4 *a, Matrix4 *b, Matrix4 *c)
/*-----------------------------------------------------*/
/* Multiplie together matrices c = ab 
   note that can be either of the input matrices */
{
  int i, j, k;
  Matrix4 *c_tmp;

  if((a==c) || (b==c))
    c_tmp = (Matrix4 *) malloc(sizeof(Matrix4));
  else
    c_tmp = c;

  for (i=0; i < 4; i++) {
    for (j=0; j < 4; j++) {
      c_tmp->elem[i][j]=0.0;
      for(k=0; k < 4; k++)
	c_tmp->elem[i][j] += a->elem[i][k] * b->elem[k][j];
    }
  }

  if((a==c) || (b==c))
    CopyMatrix(c, c_tmp);
}

/*------------------------------------------------*/
void CopyMatrix(Matrix4 *Mdest, Matrix4 *Msource)
/*------------------------------------------------*/
/* Recopie Msource dans Mdest */
{
  int i, j;

    for (i=0; i < 4; i++) 
      for (j=0; j < 4; j++) 
	Mdest->elem[i][j] = Msource->elem[i][j];
}

/*-----------------------------------*/
void SetMatrixToIdentity(Matrix4 *M)
/*-----------------------------------*/
/* Initialise M = matrice Identite */
{
  int i, j;

  for(i=0; i < 4; i++)
    for(j=0; j < 4; j++)
      M->elem[i][j] = 0.0;

  M->elem[0][0] = M->elem[1][1] = M->elem[2][2] = M->elem[3][3] = 1.0;
}

/*---------------------------------------------------------*/
void GetTransposedMatrix(Matrix4 *M, Matrix4 *Mtransposed)
/*---------------------------------------------------------*/
/* Retourne Mtransposed = transposee de M. Mtransposed doit avoir ete allouee 
   avant l'appel de cette fonction */
{
  int i, j;

  for(i = 0; i < 4; i++)
    for (j=0; j < 4; j++)
      Mtransposed->elem[j][i] = M->elem[i][j];
}


/*----------------------------------------*/
void GetTranslationMatrix(double tx, double ty, double tz, 
			  Matrix4 *M, Matrix4 *M_INV)
/*----------------------------------------*/
/* retourne la matrice de translation M correspondant a une translation 
   (tx, ty, tz) et M_INV son inverse, correspondant a une translation 
   (-tx, -ty, -tz) */
{
  SetMatrixToIdentity(M);
  SetMatrixToIdentity(M_INV);

  M->elem[0][3] = tx;
  M->elem[1][3] = ty;
  M->elem[2][3] = tz;

  M_INV->elem[0][3] = -tx;
  M_INV->elem[1][3] = -ty;
  M_INV->elem[2][3] = -tz;
}

/*---------------------------------------------------------------*/
void GetRotationMatrixX(double theta, Matrix4 *M, Matrix4 *M_INV)
/*---------------------------------------------------------------*/
/* retourne M, la matrice de Rotation en X correspondant a une rotation 
   d'angle theta, et M_INV son inverse, correspondant a une translation
   d'angle -theta (theta en degres) */
{
  SetMatrixToIdentity(M);
  SetMatrixToIdentity(M_INV);

  theta_radians = theta*M_PI/180.0;
  cos_theta = cos(theta_radians);
  sin_theta = sin(theta_radians);

  M->elem[1][1] = M->elem[2][2] = cos_theta;
  M->elem[1][2] = - sin_theta;
  M->elem[2][1] = sin_theta;

  M_INV->elem[1][1] = M_INV->elem[2][2] = cos_theta;
  M_INV->elem[1][2] = sin_theta;
  M_INV->elem[2][1] = -sin_theta;
}

/*---------------------------------------------------------------*/
void GetRotationMatrixY(double theta, Matrix4 *M, Matrix4 *M_INV)
/*---------------------------------------------------------------*/
/* retourne M, la matrice de rotation en Y correspondant a une rotation 
   d'angle theta, et M_INV son inverse, correspondant a une rotation
   d'angle -theta (theta en degres) */
{
  SetMatrixToIdentity(M);
  SetMatrixToIdentity(M_INV);

  theta_radians = theta*M_PI/180.0;
  cos_theta = cos(theta_radians);
  sin_theta = sin(theta_radians);
  
  M->elem[0][0] = M->elem[2][2] = cos_theta;
  M->elem[2][0] = - sin_theta;
  M->elem[0][2] = sin_theta;

  M_INV->elem[0][0] = M_INV->elem[2][2] = cos_theta;
  M_INV->elem[2][0] = sin_theta;
  M_INV->elem[0][2] = -sin_theta;
}

/*-----------------------------------------------------------------*/
void GetRotationMatrixZ(double theta, Matrix4 *M, Matrix4 *M_INV)
/*-----------------------------------------------------------------*/
/* retourne M, la matrice de rotation en Z correspondant a une rotation 
   d'angle theta, et M_INV son inverse, correspondant a une rotation
   d'angle -theta (theta en degres) */
{
  SetMatrixToIdentity(M);
  SetMatrixToIdentity(M_INV);
  
  theta_radians = theta*M_PI/180.0;
  cos_theta = cos(theta_radians);
  sin_theta = sin(theta_radians);

  M->elem[0][0] = M->elem[1][1] = cos_theta;
  M->elem[0][1] = -sin_theta;
  M->elem[1][0] = sin_theta;

  M_INV->elem[0][0] = M_INV->elem[1][1] = cos_theta;
  M_INV->elem[0][1] = sin_theta;
  M_INV->elem[1][0] = -sin_theta;
}

/*---------------------------------------------------------------------*/
void GetRotationMatrixXYZ(double thetax, double thetay, double thetaz, 
			  Matrix4 *M, Matrix4 *M_INV)
/*---------------------------------------------------------------------*/
/* retourne M, la matrice de Rotation en X,Y,Z correspondant a une rotation 
   d'angles (thetax, thetay, thetaz) et M_INV son inverse, correspondant a 
   une rotation d'angles (-thetax, -thetay, -thetaz) (en degres) */
{
  Matrix4 M1, M2, M_INV1, M_INV2;

  GetRotationMatrixX(thetax, &M1, &M_INV1);
  GetRotationMatrixY(thetay, &M2, &M_INV2);
  MultiplyMatrix(&M1, &M2, &M2);
  MultiplyMatrix(&M_INV1, &M_INV2, &M_INV2);

  GetRotationMatrixZ(thetaz, &M1, &M_INV1);
  MultiplyMatrix(&M1, &M2, M);
  MultiplyMatrix(&M_INV1, &M_INV2, M_INV);
}

/*-----------------------------------------------------*/
void DiffPoint3D (COORD3D *u, COORD3D *v, COORD3D *res)
/*-----------------------------------------------------*/
/* res is the difference u-v */
{
  res->x = u->x - v->x;
  res->y = u->y - v->y;
  res->z = u->z - v->z;
}

/*-----------------------------------------------------*/
void AddPoint3D (COORD3D *u, COORD3D *v, COORD3D *res)
/*-----------------------------------------------------*/
/* res is the difference u-v */
{
  res->x = u->x + v->x;
  res->y = u->y + v->y;
  res->z = u->z + v->z;
}

/*--------------------------------------------------------*/
void CrossProdPoint3D (COORD3D *u, COORD3D *v, COORD3D *w)
/*--------------------------------------------------------*/
/* stores in w the crossproduct uxv */
{
  double t1, t2, t3;
    
  t1 = u->y * v->z - u->z * v->y;
  t2 = u->z * v->x - u->x * v->z;
  t3 = u->x * v->y - u->y * v->x;

  w->x = t1;
  w->y = t2;
  w->z = t3;
}

/*---------------------------------------------*/
double DotProdPoint3D (COORD3D *u, COORD3D *v)
/*---------------------------------------------*/
/* returns the dotproduct u.v */
{
  return (u->x * v->x + u->y * v->y + u->z * v->z);
}

/*-------------------------------------------------------*/
void ScalMulPoint3D (double s, COORD3D *p, COORD3D *pres)
/*-------------------------------------------------------*/
/* pres is the product of scalar s by point p. Note that pres and p can be 
   identical */
{
    pres->x = s * p->x;
    pres->y = s * p->y;
    pres->z = s * p->z;   
}

/*------------------------------*/
double NormePoint3D (COORD3D *p)
/*------------------------------*/
/* returns the module of a vector */
{
  return ((double)sqrt (p->x * p->x + p->y * p->y + p->z * p->z));
}

/*--------------------------------*/
void NormalizeVector3D(COORD3D *u)
/*--------------------------------*/
{
  double norme = NormePoint3D(u);
  
  u->x /= norme;
  u->y /= norme;
  u->z /= norme;
}

/*--------------------------------------------------------------*/
void TransformPoint(COORD3D *ptold, Matrix4 *M, COORD3D *ptnew)
/*--------------------------------------------------------------*/
/* Calcule ptnew = M.ptold. ptnew a sa 4eme coordonnee a 1 
 ptold et ptnew sont consideres comme des points n'etant pas a l'infini */
{
  double x, y, z, w;

  x = ptold->x;
  y = ptold->y;
  z = ptold->z;

  ptnew->x = M->elem[0][0]*x + M->elem[0][1]*y + M->elem[0][2]*z + 
    M->elem[0][3];

  ptnew->y = M->elem[1][0]*x + M->elem[1][1]*y + M->elem[1][2]*z + 
    M->elem[1][3];

  ptnew->z = M->elem[2][0]*x + M->elem[2][1]*y + M->elem[2][2]*z + 
    M->elem[2][3];

  w = M->elem[3][0]*x + M->elem[3][1]*y + M->elem[3][2]*z + 
    M->elem[3][3];

  ptnew->x /= w;
  ptnew->y /= w;
  ptnew->z /= w;
}

/*------------------------------------------------------------*/
void TransformVector(COORD3D *vold, Matrix4 *M, COORD3D *vnew)
/*------------------------------------------------------------*/
/* calcule vnew = M.vold. 
 vold et vnew sont des points a l'infini (on considere leur 4eme coordonnee 
 egale a 0 */
{
  double x, y, z;

  x = vold->x;
  y = vold->y;
  z = vold->z;

  vnew->x = M->elem[0][0]*x + M->elem[0][1]*y + M->elem[0][2]*z;

  vnew->y = M->elem[1][0]*x + M->elem[1][1]*y + M->elem[1][2]*z;

  vnew->z = M->elem[2][0]*x + M->elem[2][1]*y + M->elem[2][2]*z;
}

