#include <X11/Xlib.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "object.h"
#include "graphics.h"
#include "param.h"
#include "light.h"

#include <X11/extensions/XShm.h>
#include <sys/shm.h>

#include "Tracage.h"

#define SIGN(a) (((a) < 0) ? -1 : ((a==0) ? 0 : 1))

#define AUCUN 0
#define ROTATION 1
#define TRANSLATION 2
#define ZOOM 3

#define TRUE 1
#define FALSE 0

// Structure pour le Tri des faces
typedef struct {
  int ind;
  double Point;
  double ProdVect;
} SortFaces;


/* Forward declarations */
static void mainloop();
static void get_GC_black_and_white(Window win, GC *gc);
static void get_GC_color(Window win, GC *gc);
static void reset_GC_Colors();
static void clear_double_buffer();
static void display_double_buffer();
static void DrawFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D);
static void FillFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D, short color);
static void draw_object(Object *obj, Camera *cam, Light *lum);
static void TransformSommets(Object *obj, Camera *cam);
static void close_graphics(Camera *cam);
static void AllocateBuffers(Camera *cam);

static void TraceWIREFRAME(Object *obj);
static void TraceHIDDEN_LINES(Object *obj, Camera *cam);
static void TraceFLAT_SHADING(Object *obj, Camera *cam, Light *lum);
static void TraceGOURAUD(Object *obj, Camera *cam, Light *lum);
static void TraceTRANSPARENT(Object *obj, Camera *cam, Light *lum);


/* Declarer une variable du type Display */
Display *display;

/* Specifier le nom du serveur X auquel on veut se connecter NULL
   signifie que l'on veut se connecter sur le serveur precise dans la
   variable d'environnement DISPLAY */
char *display_name=NULL;            

/* Declarer une variable contenant le numero d'ecran (screen) */
int screen;
Window win;                          /* descripteur de la fenetre */
Pixmap db;                           /* le double buffer */
GC gc, gcbg;                               /* ID du contexte graphique */
int width, height;                   /* taille de la fenetre */
int depth;
Colormap cmap;
long NColors;
int black, white;
Window root;
Visual *visual;
XImage *Image;
long Shm_Hard_Buffer_Flag;     /* flag et variable utiles pour l'allocation de Image 
                                        en shared memory */
XShmSegmentInfo Shminfo;

extern Object cube;
extern Camera cam;
extern Light lum;
extern GlobalParams params;



/*---------------------------------------*/
static void AllocateBuffers(Camera *cam)
/*---------------------------------------*/
     /* Allocation des buffers de visualisation. Normalement, pour une scene 
       complete, il faudrait faire ca pour chaque camera.*/
{
  /* Pour la shared memory X11 */
  int CompletionType; 
  long Shm_OK;
  
  /* Allocation du double buffer pour les modes de rendus simples */
  db = XCreatePixmap(display, root, cam->resX, cam->resY, depth);
  
  /* Allocation du buffer cam->image, en Shared Memory X11 si possible */
  Shm_OK = XShmQueryExtension(display);
  
  switch(Shm_OK) {
  case 0:  
    /* On a pas pas pu allouer de Shared memory. On alloue donc le Z-Buffer 
       manuellement a l'aide d'un malloc */
    /* Allocation d'un buffer 8 bits */
    cam->image = (char *) malloc(cam->resX * cam->resY * sizeof(char));
    
    if (!cam->image) {
      perror("L'allocation de cam->image a echouee!\n");
      exit(0);
    }
    
    /* Allocation d'une XImage 8 bits */
    Image = XCreateImage(display, visual, 8, ZPixmap , 0, 
			 cam->image, cam->resX, cam->resY, 
			 8, 0);
    
    /* Flag qui indique qu'on a pas alloue la Shared Memory */
    Shm_Hard_Buffer_Flag = False;
    printf("Je n'ai pas reussi a allouer la Shared memory X11!\n");
    break;
  case 1: 
    /* On a pu allouer la Shared memory. Le Z-Buffer va donc se trouver dans 
       cette zone */
    Image = XShmCreateImage(display, visual ,8, ZPixmap , 
			    NULL, &Shminfo, cam->resX, cam->resY);
    
    Shminfo.shmid = shmget(IPC_PRIVATE, 
			   cam->resX * cam->resY * sizeof(char), 
			   IPC_CREAT | 0777);

    Shminfo.shmaddr = Image->data = cam->image = 
      shmat(Shminfo.shmid, NULL, 0);
    
    if (!cam->image) {
      printf("Je n'ai pas reussi a allouer la Shared memory X11!\n");
      exit(0);
          }
    else
      printf("Shared memory X11 allouee!\n");
    
    Shminfo.readOnly = False;
    XShmAttach(display, &Shminfo);
    
    Shm_Hard_Buffer_Flag = True;
    CompletionType = XShmGetEventBase(display) + ShmCompletion; 
    break;  
  }
}


/*----------------------------------------*/
void init_graphics(int Wsizex, int Wsizey)
/*----------------------------------------*/
/* Ouvre une fenetre graphique de taille Wsizex, Wsizey */
{
  unsigned int border_width=4;         /* largeur de la bordure */
  XSetWindowAttributes WAttribs;
  unsigned long WMask;

  if((display=XOpenDisplay(display_name))==NULL) {
    fprintf(stderr,"fenetre de base: ne peut pas se connecter ");
    fprintf(stderr,"au serveur %s\n",XDisplayName(display_name));
    exit(-1);
  }

  /* Initialiser la taille de la fenetre */
  width  = Wsizex;
  height = Wsizey;

  /* Recuperer les informations concernant l'ecran (root window) sur
     lequel on va afficher notre application. Le numero de l'ecran
     par defaut s'obtient a l'aide de la macro DefaultScreen() */
    
  screen = DefaultScreen(display);
  root   = RootWindow(display, screen);
  depth  = XDefaultDepth(display, screen);
  visual = DefaultVisual(display, screen);
  black  = BlackPixel(display, screen);
  white  = WhitePixel(display, screen);

  /* Allocation du double buffer */
  db = XCreatePixmap(display, root, width, height, depth);

  //XMapWindow(display,win);

  /* Allocation du pixmap pour la visu fil de fer et flat shading, et d'une 
     XImage pour la visu gouraud, phong et textures */
//  AllocateBuffers(&cam);

  if(params.color_mode == BLACK_AND_WHITE_MODE) {
    /* creation d'une fenetre opaque a l'aide de la fonction
       XCreateSimpleWindow(). L'appel de XCreateSimpleWindow au lieu de
       XCreateWindow fait que la fenetre cree herite de tous les
       attributs de la fenetre parent */

    win = XCreateSimpleWindow(display, root, 0, 0,
			      width, height, border_width,
			      black, white);

    get_GC_black_and_white(win, &gc);
  }
  else {
    /* COLOR MODE */
    printf("Color mode on!\n");
    /* creation d'une colormap privee en niveaux de gris */
    cmap   = XCreateColormap(display, root, visual, AllocAll);
    Lut_GreyLevels(display, cmap); 
    //Lut_16M(display, cmap);

    WAttribs.save_under = True;
    WAttribs.border_pixel = 0;
    WAttribs.colormap = cmap; 
    WAttribs.override_redirect = False;
    WAttribs.backing_store = Always;
	        
    WMask= CWBorderPixel | CWBackingStore | CWSaveUnder | 
      CWOverrideRedirect | CWColormap ;

    win=XCreateWindow(display, 
		      root, 
		      0, 0, 
		      (unsigned int) width, (unsigned int) height,
		      0,
		      CopyFromParent,
		      CopyFromParent,
		      CopyFromParent,
		      WMask,
		      &WAttribs);

    get_GC_color(win, &gc);
  }

  /* selection des evenements que l'on desirera traiter pour cette fenetre */
  XSelectInput(display, win, ExposureMask | KeyPressMask |
	       ButtonPressMask | ButtonReleaseMask | StructureNotifyMask | 
	       PointerMotionMask);


  /* Pour le double buffer */
  gcbg = XCreateGC(display, root, 0, NULL);
  XSetForeground(display, gcbg, white);

  /* affichage (mapping) de la fenetre, on mappe les differentes
     fenetres juste avant la gestion des evenements */

  XMapWindow(display,win);

  /* Allocation du pixmap pour la visu fil de fer et flat shading, et d'une 
     XImage pour la visu gouraud, phong et textures */
  AllocateBuffers(&cam);

  mainloop();

} 

void BouleVirtuelle(COORD3D *v1, COORD3D *v2, Matrix4 *M, Matrix4 *InvM) {
  // Cree les matrices de la boule virtuelle
  // v1 et v2 doivent etre normalise
  COORD3D w, v;
  double angle;
  Matrix4 Rot, InvRot, Rz;

  // Calcul du repere
  CrossProdPoint3D(v1, v2, &w);
  NormalizeVector3D(&w);
  CrossProdPoint3D(&w, v1, &v);
  // Calcul de la matrice d'alignement
  SetMatrixToIdentity(&Rot);
  Rot.elem[0][0] = v1->x;
  Rot.elem[0][1] = v1->y;
  Rot.elem[0][2] = v1->z;
  Rot.elem[1][0] = v.x;
  Rot.elem[1][1] = v.y;
  Rot.elem[1][2] = v.z;
  Rot.elem[2][0] = w.x;
  Rot.elem[2][1] = w.y;
  Rot.elem[2][2] = w.z;
  GetTransposedMatrix(&Rot, &InvRot);
  // Calcul de la matrice de rotation
  angle = -(180./M_PI)*acos(DotProdPoint3D(v1, v2));
  GetRotationMatrixZ(angle, &Rz, InvM);
  // Calcul de la matrice
  MultiplyMatrix(&Rz, &Rot, M);
  MultiplyMatrix(&InvRot, M, M);
  GetTransposedMatrix(M, InvM);
}


/*----------------------*/
static void mainloop()
/*----------------------*/
/* gestion des evenements:
   -prend un evenement
   -utilise le premier expose pour afficher texte et graphiques
   */
{
  XEvent report;                      /* structure des evenements */
  int old_x,old_y,new_x,new_y,mode_deplacement;
  int dx,dy;
  double tx, ty, tz;
  COORD3D v1, v2;
  double R;
  Matrix4 M, InvM;


  params.render_mode = WIREFRAME;
  params.eclairage = DEUXFRANCS;
  params.select_object = OBJECT;
  cam.use_pixmap = TRUE;
  draw_object(&cube, &cam, &lum);
  while(1) {
    XNextEvent(display, &report);
    switch (report.type) {
    case Expose:
      /* on purge la queue des evenements de tous les Expose, pour
	 ne redessiner la fenetre qu'une seule fois */
  
      while(XCheckTypedEvent(display, Expose, &report));
      draw_object(&cube, &cam, &lum);
      break;

    case ConfigureNotify:
      break;

    case ButtonPress:
      old_x = report.xbutton.x;
      old_y = report.xbutton.y;

      switch(report.xbutton.button) {
      case Button1 :
	mode_deplacement = ROTATION;
	break;
       
      case Button2:
	mode_deplacement = TRANSLATION;
	break;
       
      case Button3 :
	mode_deplacement = ZOOM;
	break;
      }
      break;


    case ButtonRelease:
      mode_deplacement = AUCUN;
      break;


    case KeyPress: {
      /* Les 4 lignes suivantes sont necessaires sous X11 pour detecter la 
	 valeur d'une touche entree au clavier dans une fenetre */
	char buffer;
	int bufsize=1;
	int charcount;

	charcount = XLookupString(&report, &buffer, bufsize, NULL, 
				  NULL);

	switch(buffer) {
	case 'c':
	  // Place la camera
	  PlacerCamera(&cam,&cube);
	  UpdateLocalToCamera(&cube, &cam);
	  draw_object(&cube, &cam, &lum);
	  break;
	case 'q':
	  /* Quitte le programme */
	  close_graphics(&cam);
	  exit(1);
	  break;
	case 'w':
	  /* Visualisation fil de fer */
	  params.render_mode = WIREFRAME;
	  cam.render_mode = WIREFRAME;
	  cam.use_pixmap = TRUE;
	  reset_GC_Colors();
	  draw_object(&cube, &cam, &lum);
	  printf("Render mode = WIREFRAME\n");
	  break;
	case 'l':
	  /* Visualisation avec les faces cachees */
	  cam.render_mode = HIDDEN_LINES;
	  cam.use_pixmap = TRUE;
	  params.render_mode = HIDDEN_LINES;
	  reset_GC_Colors();
	  draw_object(&cube, &cam, &lum);
	  printf("Render mode = HIDDEN_LINES\n");
	  break;
	case 'f':
	  /* Visualisation en flat shading (eclairages) */
	  params.render_mode = FLAT_SHADING;
	  cam.render_mode = FLAT_SHADING;
	  cam.use_pixmap = TRUE;
	  params.eclairage = DEUXFRANCS;
	  reset_GC_Colors();
	  draw_object(&cube, &cam, &lum);
	  printf("Render mode = FLAT_SHADING\n");
	  break;
	case 'g':
	  /* Visualisation en gouraud */
	  params.render_mode = GOURAUD;
	  cam.render_mode = GOURAUD;
	  cam.use_pixmap = FALSE;
	  params.eclairage = NORMAL;
	  reset_GC_Colors();
	  // Place la lumiere sur la camera
	  tx = cam.LocalToGlobal.elem[0][3] - lum.LocalToGlobal.elem[0][3];
	  ty = cam.LocalToGlobal.elem[1][3] - lum.LocalToGlobal.elem[1][3];
	  tz = cam.LocalToGlobal.elem[2][3] - lum.LocalToGlobal.elem[2][3];
	  TranslateLight(&lum, tx, ty, tz);
	  draw_object(&cube, &cam, &lum);
	  printf("Render mode = GOURAUD\n");
	  break;
	case 't':
	  /* Visualisation en transparence */
	  params.render_mode = TRANSPARENT;
	  cam.render_mode = TRANSPARENT;
	  cam.use_pixmap = FALSE;
	  params.eclairage = NORMAL;
	  reset_GC_Colors();
	  // Place la lumiere sur la camera
	  tx = cam.LocalToGlobal.elem[0][3] - lum.LocalToGlobal.elem[0][3];
	  ty = cam.LocalToGlobal.elem[1][3] - lum.LocalToGlobal.elem[1][3];
	  tz = cam.LocalToGlobal.elem[2][3] - lum.LocalToGlobal.elem[2][3];
	  TranslateLight(&lum, tx, ty, tz);
	  draw_object(&cube, &cam, &lum);
	  printf("Render mode = TRANSPARENCE\n");
	  break;
	case 'o' :
	  if (params.select_object==OBJECT) {
	    params.select_object=LIGHT;
	    printf("Lumiere selectionne\n");
	  }
	  else {
	    params.select_object=OBJECT;
	    printf("Objet selectionnee\n");
	  }
	  break;
	case 'e' :
	  if (params.eclairage==DEUXFRANCS) {
	    params.eclairage = NORMAL;
	    printf("Eclairage normal\n");
	    // Place la lumiere sur la camera
	    tx = cam.LocalToGlobal.elem[0][3] - lum.LocalToGlobal.elem[0][3];
	    ty = cam.LocalToGlobal.elem[1][3] - lum.LocalToGlobal.elem[1][3];
	    tz = cam.LocalToGlobal.elem[2][3] - lum.LocalToGlobal.elem[2][3];
	    TranslateLight(&lum, tx, ty, tz);
	  }
	  else {
	    params.eclairage = DEUXFRANCS;
	    printf("Eclairage a 2 Frs\n");
	  }
	  draw_object(&cube, &cam, &lum);
	  break;

	case '1':
	  if (params.eclairage==NORMAL) {
	    if (cube.material.Ka<=0.9)
	      cube.material.Ka += 0.1;
	    printf("Ka = %g\n",cube.material.Ka);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '2':
	  if (params.eclairage==NORMAL) {
	    if (cube.material.Ka>=0.1)
	      cube.material.Ka -= 0.1;
	    printf("Ka = %g\n",cube.material.Ka);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '3':
	  if (params.eclairage==NORMAL) {
	    cube.material.Ka = 0.3;
	    printf("Ka = %g\n",cube.material.Ka);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '4':
	  if (params.eclairage==NORMAL) {
	    if (cube.material.Kd<=0.9)
	      cube.material.Kd += 0.1;
	    printf("Kd = %g\n",cube.material.Kd);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '5':
	  if (params.eclairage==NORMAL) {
	    if (cube.material.Kd>=0.1)
	      cube.material.Kd -= 0.1;
	    printf("Kd = %g\n",cube.material.Kd);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '6':
	  if (params.eclairage==NORMAL) {
	    cube.material.Kd = 0.3;
	    printf("Kd = %g\n",cube.material.Kd);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '7':
	  if (params.eclairage==NORMAL) {
	    if (cube.material.Ks<=0.9)
	      cube.material.Ks += 0.1;
	    printf("Ks = %g\n",cube.material.Ks);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '8':
	  if (params.eclairage==NORMAL) {
	    if (cube.material.Ks>=0.1)
	      cube.material.Ks -= 0.1;
	    printf("Ks = %g\n",cube.material.Ks);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '9':
	  if (params.eclairage==NORMAL) {
	    cube.material.Ks = 0.3;
	    printf("Ks = %g\n",cube.material.Ks);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '+':
	  if (params.eclairage==NORMAL) {
	    ++cube.material.m;
	    printf("m = %d\n",cube.material.m);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '-':
	  if (params.eclairage==NORMAL) {
	    if (cube.material.m>0)
	      --cube.material.m;
	    printf("m = %d\n",cube.material.m);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
	case '*':
	  if (params.eclairage==NORMAL) {
	    cube.material.m = 20;
	    printf("m = %d\n",cube.material.m);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;
     
	}
      }
      break;

    case MotionNotify:
      while(XCheckTypedEvent(display, MotionNotify, &report));

      new_x = report.xmotion.x;
      new_y = report.xmotion.y;

      /* dplacement en pixels */
      dx = new_x - old_x;
      dy = new_y - old_y;

      switch(mode_deplacement) {
        case TRANSLATION:
	  if (params.select_object==OBJECT) {
	    tx = (3.*cube.boundingBox.plusGrandCote*dx)/(cam.resX);
	    ty = (3.*cube.boundingBox.plusGrandCote*dy)/(cam.resY);
	    
	    TranslateObject(&cube, tx, -ty, 0.);
	    UpdateLocalToCamera(&cube, &cam);
	    draw_object(&cube, &cam, &lum);
	  }
	  else {
	    tx = (3.*dx)/(cam.resX);
	    ty = (3.*dy)/(cam.resY);
	    
	    TranslateLight(&lum, tx, -ty, 0.);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;

        case ROTATION:
	  // Construction de v1 & v2
	  v1.x = old_x - cam.tx;
	  v1.y = old_y - cam.ty;
	  v2.x = new_x - cam.tx;
	  v2.y = new_y - cam.ty;
	  if (cam.tx<cam.ty)
	    R = cam.tx*cam.tx;
	  else R = cam.ty*cam.ty;
	  v1.z = (v1.x*v1.x+v1.y*v1.y>R)?0:-sqrt(R - v1.x*v1.x - v1.y*v1.y);
	  v2.z = (v2.x*v2.x+v2.y*v2.y>R)?0:-sqrt(R - v2.x*v2.x - v2.y*v2.y);
	  NormalizeVector3D(&v1);
	  NormalizeVector3D(&v2);
	  BouleVirtuelle(&v1, &v2, &M, &InvM);
	  if (params.select_object==OBJECT) {
	    // Modification de l'objet
	    tx = cube.LocalToGlobal.elem[0][3];
	    ty = cube.LocalToGlobal.elem[1][3];
	    tz = cube.LocalToGlobal.elem[2][3];
	    TranslateObject(&cube, -tx, -ty, -tz);
	    MultiplyMatrix(&M, &cube.LocalToGlobal, &cube.LocalToGlobal);
	    MultiplyMatrix(&cube.GlobalToLocal, &InvM, &cube.GlobalToLocal);
	    TranslateObject(&cube, tx, ty, tz);
	    UpdateLocalToCamera(&cube, &cam);
	    draw_object(&cube, &cam, &lum);
	  }
	  else {
	    // Modification de la lumiere
	    tx = cube.LocalToGlobal.elem[0][3];
	    ty = cube.LocalToGlobal.elem[1][3];
	    tz = cube.LocalToGlobal.elem[2][3];
	    TranslateLight(&lum, -tx, -ty, -tz);
	    MultiplyMatrix(&M, &lum.LocalToGlobal, &lum.LocalToGlobal);
	    MultiplyMatrix(&lum.GlobalToLocal, &InvM, &lum.GlobalToLocal);
	    TranslateLight(&lum, tx, ty, tz);
	    draw_object(&cube, &cam, &lum);
	  }
	  break;

        case ZOOM:
	  tz = (3.*cube.boundingBox.plusGrandCote*dx)/cam.resX;
	  TranslateObject(&cube, 0., 0., tz);
	  UpdateLocalToCamera(&cube, &cam);
	  draw_object(&cube, &cam, &lum);
	  break;
        }
        old_x = new_x;
        old_y = new_y;
      break;

    default:
      /* tous les evenements selectionnes par StructureNotifyMask 
	 (DestroyNotify, GravityNotify, etc...) sont filtres ici. */
      break;
    } 
  } 

}



/*----------------------------------------------------*/
static void get_GC_black_and_white(Window win, GC *gc)
/*----------------------------------------------------*/
/* Configure le contexte graphique pour du dessin en noir et blan 
   (couleur du trait = noir, fond = blanc */
{
  unsigned long valuemask = 0;                /* Ignore XGCvalues et prend les 
					         valeurs par defaut */
  XGCValues values;

  /* creation d'un contexte graphique par defaut */
  *gc = XCreateGC(display, win, valuemask, &values);

  /* specification d'un foreground noir */
  XSetForeground(display, *gc, black); 
}

/*-------------------------------------------*/
static void get_GC_color(Window win, GC *gc)
/*-------------------------------------------*/
/* COnfigure le contexte graphique pour un ecran couleur. Par defaut on 
   dessine en noir sur blanc */
{
  unsigned long valuemask = 0;                /* Ignore XGCvalues et prend les 
					         valeurs par defaut */
  XGCValues values;

  /* creation d'un contexte graphique par defaut */
  *gc = XCreateGC(display, win, valuemask, &values);

  /* specification d'un foreground blanc */
  XSetForeground(display, *gc, 255); 
}

/*--------------------*/
static void reset_GC_Colors()
/*--------------------*/
/* Remet la bonne valeur pour la couleur du dessin (noir en mode N/B, blanc 
   en mode couleur) */
{
  if(params.color_mode == BLACK_AND_WHITE_MODE)
    XSetForeground(display, gc, black);
  else
    XSetForeground(display, gc, 255); 
}

/*------------------------*/
static void clear_double_buffer()
/*------------------------*/
/* Efface le contenu du double buffer */
{
  int y; 
  
  if(cam.use_pixmap) 
    XFillRectangle(display, db, gcbg, 0, 0, cam.resX, cam.resY); 
  else 
    /* la couleur. 0 = ecran noir */ 
    cam.image = (char *) memset(cam.image, 0, sizeof(char) * 
				 cam.resX * cam.resY); 
}

/*---------------------------------*/
static void display_double_buffer()
/*---------------------------------*/
/* Recopie le double buffer dans la fenetre graphique */
{
  if(cam.use_pixmap) 
    XCopyArea(display, db, win, gc, 0, 0, cam.resX, cam.resY, 0, 0); 
  else { 
    if(Shm_Hard_Buffer_Flag) { 
      XShmPutImage(display,win,gc,Image,0,0,0,0,cam.resX, cam.resY,NULL); 
    } 
    else { 
      XPutImage(display,win,gc,Image,0,0,0,0,cam.resX, cam.resY); 
    } 
  } 
  XFlush(display);
}

/*----------------------------------------------------------*/
static void DrawFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D)
/*----------------------------------------------------------*/
/* Dessine la face definie par les sommets A,B,C,D, en mode ligne.
   Si C == D alors il s'agit d'un triangle */
{
  XPoint point[5];
  
  point[0].x = A->x;
  point[0].y = A->y;
  point[1].x = B->x;
  point[1].y = B->y;
  point[2].x = C->x;
  point[2].y = C->y;

  if(C != D) {
    point[3].x = D->x;
    point[3].y = D->y;
    point[4].x = A->x;
    point[4].y = A->y;

    XDrawLines(display, db, gc, point, 5, CoordModeOrigin);
  }
  else {
    point[3].x = A->x;
    point[3].y = A->y;

    XDrawLines(display, db, gc, point, 4, CoordModeOrigin);
  }
}

/*-----------------------------------------------------------------------*/
static void FillFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D, short color)
/*-----------------------------------------------------------------------*/
/* Dessine la face definie par les sommets A,B,C,D, en mode plein.
   Si C == D alors il s'agit d'un triangle */
{
  XPoint point[4];
  
  XSetForeground(display, gc, color);

  point[0].x = A->x;
  point[0].y = A->y;
  point[1].x = B->x;
  point[1].y = B->y;

  if(C != D) {
    point[2].x = C->x;
    point[2].y = C->y;
    point[3].x = D->x;
    point[3].y = D->y;

    XFillPolygon(display, db, gc, point, 4, Convex, CoordModeOrigin);
  }
  else {
    point[2].x = C->x;
    point[2].y = C->y;

    XFillPolygon(display, db, gc, point, 3, Convex, CoordModeOrigin);
  }
}


static int ZCompare(const SortFaces *i, const SortFaces *j) {
	if (i->Point>j->Point)
		return 1;
	else if (i->Point<j->Point)
		return -1;
	return 0;
}


/*-----------------------------------------------*/
static void draw_object(Object *obj, Camera *cam, Light *lum)
/*-----------------------------------------------*/
/* Affiche dans la fenetre graphique l'objet passe en 1er parametre, comme 
   s'il etait vu par la camera passe en 2eme parametre */
{

  /* 1) On efface le contenu du double buffer */
  clear_double_buffer(); 

  /* 2) On met a jour les champs CameraVertex et ScreenVertex des sommets de
     l'objet pour pouvoir les dessiner */
  TransformSommets(obj, cam);

  switch ( params.render_mode) {
  case WIREFRAME:
    TraceWIREFRAME(obj);
    break;
  case HIDDEN_LINES:
    TraceHIDDEN_LINES(obj, cam);
    break;
  case FLAT_SHADING:
    TraceFLAT_SHADING(obj, cam, lum);
    break;
  case GOURAUD:
    TraceGOURAUD(obj, cam, lum);
    break;
  case TRANSPARENT:
    TraceTRANSPARENT(obj, cam, lum);
    break;
  default:
    break;
  }

  /* 4) Recopie le double buffer dans la fenetre X-Window */
  display_double_buffer();
}

/*------------------------------------------------------*/
static void TransformSommets(Object *obj, Camera *cam)
/*------------------------------------------------------*/
/* met a jour les champs CameraVertex et ScreenVertex de tous les sommets de 
   l'objet passe en 1er parametre */
{
     /* 1) CODE A ECRIRE : 

        Pour chaque sommet, projeter, ranger dans le champ CameraVertex, puis
        mise a l'echelle et translation  pour passer des 
	unites utilisees dans le repere du monde aux unites X11 (pixels). 
	Translation car le repere X11 n'est pas centre au centre de la 
	fenetre de visu , il est centre en haut a gauche de la fenetre de 
	visualisation, X a droite, Y en bas 

        Fonctions a utiliser : TransformPoint() */
 

    /*  2) CODE A ECRIRE : 

       Projection et mise a l'echelle et translation  pour passer des 
       unites utilisees dans le repere du monde aux unites X11 (pixels). 
       Translation car le repere X11 n'est pas centre au centre de la 
       fenetre de visu , il est centre en haut a gauche de la fenetre de 
       visualisation, X a droite, Y en bas */
  int i;

  //UpdateLocalToCamera(obj,cam);
  for (i=0; i<(obj->NbVertices); ++i) {
    TransformPoint(&(obj->Vertex[i].ObjectVertex), &(obj->LocalToCamera), &(obj->Vertex[i].CameraVertex));
    obj->Vertex[i].ScreenVertex.x = ((obj->Vertex[i].CameraVertex.x/obj->Vertex[i].CameraVertex.z)*cam->sx)+cam->tx;
    obj->Vertex[i].ScreenVertex.y = ((obj->Vertex[i].CameraVertex.y/obj->Vertex[i].CameraVertex.z)*cam->sy)+cam->ty;
  }
}


/*------------------------*/
void close_graphics(Camera *cam)
/*------------------------*/
     /* Libere les ressources allouees */
{    
  switch (Shm_Hard_Buffer_Flag) {
  case True: {
    printf("Liberation de la memoire partagee\n");
    XShmDetach(display, &Shminfo);
    XDestroyImage(Image);
    shmdt(Shminfo.shmaddr);
    shmctl(Shminfo.shmid, IPC_RMID, 0);
    break;
  }
  case False: {
    /* Le buffer image */
    free(cam->image);
    XDestroyImage(Image);
    break;
  }
  }
  XCloseDisplay(display); 
}


// Routine WIREFRAME
void TraceWIREFRAME(Object *obj) {
  int i0, i1, i2, i3;
  int current_tri;

  current_tri = obj->NbTri;
  while(current_tri) {
    i0 = obj->TriList[current_tri-1].PtVertex[0];
    i1 = obj->TriList[current_tri-1].PtVertex[1];
    i2 = obj->TriList[current_tri-1].PtVertex[2];
    i3=i2;
    current_tri--;
    
    DrawFace(&(obj->Vertex[i0].ScreenVertex), 
	     &(obj->Vertex[i1].ScreenVertex),
	     &(obj->Vertex[i2].ScreenVertex),
	     &(obj->Vertex[i3].ScreenVertex));
  }
}


// Routine HIDDEN_LINES
void TraceHIDDEN_LINES(Object *obj, Camera *cam) {
  int i0, i1, i2, i3, i;
  int current_tri;
  COORD3D V, N;
  SortFaces ListeFaces[obj->NbTri];
  int NbFaces = 0;
  double ProdVect;
 
  for (i=0; i<obj->NbTri; ++i) {
    // Calcul normale
    TransformVector(&(obj->TriList[i].N), &(obj->LocalToCamera), &N);
    NormalizeVector3D(&N);
    // Calcul direction
    V.x = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.x;
    V.y = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.y;
    V.z = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.z;
    NormalizeVector3D(&V);
    // Test Visibilite
    if ((ProdVect=DotProdPoint3D(&N, &V))>0) {
      ListeFaces[NbFaces].ind = i;
      ListeFaces[NbFaces].Point = (obj->Vertex[obj->TriList[i].PtVertex[0]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[2]].CameraVertex.z) / 3;
      //ListeFaces[NbFaces].Point = obj->Vertex[obj->TriList[i].PtVertex[0]].CameraVertex.z;
      ListeFaces[NbFaces].ProdVect = ProdVect;
      NbFaces++;
    }
  }
  // Tri faces
  qsort(ListeFaces, NbFaces, sizeof(SortFaces), ZCompare);
  // Affichage faces
  current_tri = NbFaces;
  while(current_tri) {
    i0 = obj->TriList[ListeFaces[current_tri-1].ind].PtVertex[0];
    i1 = obj->TriList[ListeFaces[current_tri-1].ind].PtVertex[1];
    i2 = obj->TriList[ListeFaces[current_tri-1].ind].PtVertex[2];
    i3=i2;
    DrawFace(&(obj->Vertex[i0].ScreenVertex), 
	     &(obj->Vertex[i1].ScreenVertex),
	     &(obj->Vertex[i2].ScreenVertex),
	     &(obj->Vertex[i3].ScreenVertex));
    current_tri--;
  }
}


// Routine FLAT_SHADING
void TraceFLAT_SHADING(Object *obj, Camera *cam, Light *lum) {
  int i0, i1, i2, i3, i;
  int current_tri;
  COORD3D V, N;
  SortFaces ListeFaces[obj->NbTri];
  int NbFaces = 0;
  double ProdVect;
  int Intensite;

  for (i=0; i<obj->NbTri; ++i) {
    // Calcul normale
    TransformVector(&(obj->TriList[i].N), &(obj->LocalToCamera), &N);
    NormalizeVector3D(&N);
    // Calcul direction
    V.x = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.x;
    V.y = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.y;
    V.z = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.z;
    NormalizeVector3D(&V);
    // Test Visibilite
    if ((ProdVect=DotProdPoint3D(&N, &V))>0) {
      ListeFaces[NbFaces].ind = i;
      ListeFaces[NbFaces].Point = (obj->Vertex[obj->TriList[i].PtVertex[0]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[2]].CameraVertex.z) / 3;
      //ListeFaces[NbFaces].Point = obj->Vertex[obj->TriList[i].PtVertex[0]].CameraVertex.z;
      ListeFaces[NbFaces].ProdVect = ProdVect;
      NbFaces++;
    }
  }
  // Tri faces
  qsort(ListeFaces, NbFaces, sizeof(SortFaces), ZCompare);
  // Affichage faces
  current_tri = NbFaces;
  while(current_tri) {
    i0 = obj->TriList[ListeFaces[current_tri-1].ind].PtVertex[0];
    i1 = obj->TriList[ListeFaces[current_tri-1].ind].PtVertex[1];
    i2 = obj->TriList[ListeFaces[current_tri-1].ind].PtVertex[2];
    i3=i2;
    if (params.eclairage==DEUXFRANCS) {
      // Eclairage a 2 balles
      Intensite = 255 * ListeFaces[current_tri-1].ProdVect;
    }
    else {
      // Eclairage normal
      Intensite = LumiereAmbiente(lum, obj) + LumiereDiffuse(lum, obj, ListeFaces[current_tri-1].ind, &obj->TriList[ListeFaces[current_tri-1].ind].N) + LumiereSpeculaire(lum, obj, cam, ListeFaces[current_tri-1].ind, &obj->TriList[ListeFaces[current_tri-1].ind].N);
      if (Intensite>255)
	Intensite = 255;
    }
    
    FillFace(&(obj->Vertex[i0].ScreenVertex), 
	     &(obj->Vertex[i1].ScreenVertex),
	     &(obj->Vertex[i2].ScreenVertex),
	     &(obj->Vertex[i3].ScreenVertex), Intensite);
    current_tri--;
  }
}


// Routine GOURAUD
void TraceGOURAUD(Object *obj, Camera *cam, Light *lum) {
  int i;
  int current_tri;
  COORD3D V, N;
  SortFaces ListeFaces[obj->NbTri];
  int NbFaces = 0;
  double ProdVect;
  
  for (i=0; i<obj->NbTri; ++i) {
    // Calcul normale
    TransformVector(&(obj->TriList[i].N), &(obj->LocalToCamera), &N);
    NormalizeVector3D(&N);
    // Calcul direction
    V.x = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.x;
    V.y = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.y;
    V.z = -obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.z;
    NormalizeVector3D(&V);
    // Test Visibilite
    if ((ProdVect=DotProdPoint3D(&N, &V))>0) {
      ListeFaces[NbFaces].ind = i;
      ListeFaces[NbFaces].Point = (obj->Vertex[obj->TriList[i].PtVertex[0]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[2]].CameraVertex.z) / 3;
      //ListeFaces[NbFaces].Point = obj->Vertex[obj->TriList[i].PtVertex[0]].CameraVertex.z;
      ListeFaces[NbFaces].ProdVect = ProdVect;
      NbFaces++;
    }
  }
  // Tri faces
  qsort(ListeFaces, NbFaces, sizeof(SortFaces), ZCompare);
  // Affichage faces
  current_tri = NbFaces;
  SetImageBuffer(cam);
  while(current_tri) {
    G_polygon(ListeFaces[current_tri-1].ind, obj, cam, lum);
    current_tri--;
  }
}


// Routine TRANSPARENCE
void TraceTRANSPARENT(Object *obj, Camera *cam, Light *lum) {
  int i;
  int current_tri;
  SortFaces ListeFaces[obj->NbTri];
  int NbFaces = 0;
  
  for (i=0; i<obj->NbTri; ++i) {
    ListeFaces[NbFaces].ind = i;
    ListeFaces[NbFaces].Point = (obj->Vertex[obj->TriList[i].PtVertex[0]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[1]].CameraVertex.z + obj->Vertex[obj->TriList[i].PtVertex[2]].CameraVertex.z) / 3;
    NbFaces++;
  }
  // Tri faces
  qsort(ListeFaces, NbFaces, sizeof(SortFaces), ZCompare);
  // Affichage faces
  current_tri = NbFaces;
  SetImageBuffer(cam);
  while(current_tri) {
    T_polygon(ListeFaces[current_tri-1].ind, obj, cam, lum);
    current_tri--;
  }
}
