#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <limits.h>
#include "object.h"
#include "psb_draw.h"
#include "light.h"

#define MAX(a,b) (((a)>(b)) ? (a):(b))
#define MIN(a,b) (((a)<(b)) ? (a):(b))

char *G_buffer;
int HW_SCREEN_X_SIZE;
int HW_SCREEN_Y_SIZE;
int *G_start;
int *G_end;
int G_miny,G_maxy;                          /* vertical boundaries */

double *ZBuffer;

/*-------------------------------*/
void SetImageBuffer(Camera *cam)
/*-------------------------------*/
/* Initialisation du buffer camera. DOIT ETRE APPELEE AVANT TOUTES LES AUTRES ! */
{
  G_buffer = cam->image;
  ZBuffer  = cam->zbuffer;
  HW_SCREEN_X_SIZE=cam->resX;
  HW_SCREEN_Y_SIZE=cam->resY;

   /* Allocation du tableau pour le trace des polygones */
  AllocateTableau();
}

/*----------------------------------------------------------*/
void G_line(int x,int y,int x2,int y2,unsigned char colour)
/*----------------------------------------------------------*/
/* Trace de ligne de type Bresenham, version optimisee pour l'utilisation 
   d'une XImage */
{
 int dx,dy,long_d,short_d;
 int d,add_dh,add_dl;
 int inc_xh,inc_yh,inc_xl,inc_yl;
 register int inc_ah,inc_al;
 register int i;
 register unsigned char *adr=G_buffer;

 dx=x2-x; dy=y2-y;                          /* ranges */

 if(dx<0)
 {
   dx=-dx;
   inc_xh=-1;
   inc_xl=-1;
 }    /* making sure dx and dy >0 */
 else
 {
   inc_xh=1;
   inc_xl=1;
 }    /* adjusting increments */

 if(dy<0)
 {
   dy=-dy;
   inc_yh=-HW_SCREEN_X_SIZE;
   inc_yl=-HW_SCREEN_X_SIZE;
 }    /* to get to the neighboring */
 else
 {
   inc_yh=HW_SCREEN_X_SIZE; /* point along Y have to add */
   inc_yl= HW_SCREEN_X_SIZE;
 }    /* or subtract this */

 if(dx>dy)
 {
   long_d=dx;
   short_d=dy;
   inc_yl=0;
 }                           
 else
 {
   long_d=dy;
   short_d=dx;
   inc_xl=0;
 }    /* x or y is changed in L case */

 inc_ah=inc_xh+inc_yh;
 inc_al=inc_xl+inc_yl;                      /* increments for point adress */
 adr+=y*HW_SCREEN_X_SIZE+x;                 /* address of first point */

 d=2*short_d-long_d;                        /* initial value of d */
 add_dl=2*short_d;                          /* d adjustment for H case */
 add_dh=2*short_d-2*long_d;                 /* d adjustment for L case */

 for(i=0;i<=long_d;i++)                     /* for all points in longer range */
 {
   *adr=colour;                              /* rendering */

   if(d>=0)
   {
     adr+=inc_ah;
     d+=add_dh;
   }         /* previous point was H type */
   else
   {
     adr+=inc_al;
     d+=add_dl;
   }         /* previous point was L type */
 }
}


void AllocateTableau()
{
  /* Allocation du tableau des lignes */
  G_start = (int *) malloc(HW_SCREEN_Y_SIZE * sizeof(int));
  G_end = (int *) malloc(HW_SCREEN_Y_SIZE * sizeof(int));
}


void GI_scan(int x1, int y1, int x2, int y2)
/* Balayage de l'arete et remplissage des tableaux G_start et G_end */
{
 int dx,dy,long_d,short_d;
 int d,add_dh,add_dl;
 int inc_xh,inc_yh,inc_xl,inc_yl;
 register int i; 

 dx=x2-x1; dy=y2-y1;                          /* ranges */

  if(y1<G_miny) G_miny=y1;
  if(y1>G_maxy) G_maxy=y1;
  if(y2<G_miny) G_miny=y2;
  if(y2>G_maxy) G_maxy=y2;                  /* updating vertical size */


  if(dx<0){dx=-dx; inc_xh=-1; inc_xl=-1;}   /* making sure dx and dy >0 */
  else    {        inc_xh=1;  inc_xl=1; }   /* adjusting increments */
  if(dy<0){dy=-dy; inc_yh=-1; inc_yl=-1;}
  else    {        inc_yh=1;  inc_yl=1; }

  if(dx>dy){long_d=dx;short_d=dy;inc_yl=0;} /* long range,&making sure either */
  else     {long_d=dy;short_d=dx;inc_xl=0;} /* x or y is changed in L case */

  d=2*short_d-long_d;                       /* initial value of d */
  add_dl=2*short_d;                         /* d adjustment for H case */
  add_dh=2*short_d-2*long_d;                /* d adjustment for L case */

  for(i=0;i<=long_d;i++) {                 /* for all points in longer range */
    if(x1<G_start[y1])                         /* further then rightmost */
      G_start[y1]=x1;                           /* the begining of scan line */

   if(G_end[y1]<x1)                           /* further the leftmost */
     G_end[y1]=x1;                             /* the end of scan line */

   if(d>=0){x1+=inc_xh;y1+=inc_yh;d+=add_dh;} /* previous point was H type */
   else    {x1+=inc_xl;y1+=inc_yl;d+=add_dl;} /* previous point was L type */
  }
}

void GI_boarder_array_init(void)
{
 G_miny=INT_MAX;                            /* polygon starts here */
 G_maxy=INT_MIN;                            /* polygon ends here */

 G_start = (int *) memset((char *)G_start, 127, HW_SCREEN_Y_SIZE*sizeof(char*));
 G_end =(int *)  memset((char *)G_end, 0, HW_SCREEN_Y_SIZE*sizeof(char*));
}

void G_ambient_polygon(Tri *tri, Object *obj, unsigned char colour)
{
 register unsigned char *l_adr,*adr; 
 register int beg,end;
 int i0, i1, i2;

 /* Initialisation du tableau */
 GI_boarder_array_init();                   /* initializing the arrays */


 /* Parcours des faces */
 i0 = tri->PtVertex[0];
 i1 = tri->PtVertex[1];
 i2 = tri->PtVertex[2];

 GI_scan(obj->Vertex[i0].ScreenVertex.x,
         obj->Vertex[i0].ScreenVertex.y,
         obj->Vertex[i1].ScreenVertex.x,
         obj->Vertex[i1].ScreenVertex.y); 

 GI_scan(obj->Vertex[i1].ScreenVertex.x,
         obj->Vertex[i1].ScreenVertex.y,
         obj->Vertex[i2].ScreenVertex.x,
         obj->Vertex[i2].ScreenVertex.y); 

 GI_scan(obj->Vertex[i2].ScreenVertex.x,
         obj->Vertex[i2].ScreenVertex.y,
         obj->Vertex[i0].ScreenVertex.x,
         obj->Vertex[i0].ScreenVertex.y); 


  /* nothing to do? */
 if(G_miny<=G_maxy)
 {
   l_adr=G_buffer+G_miny*HW_SCREEN_X_SIZE;   /* addr of 1st byte of 1st line */

   /* rendering all lines */
   for(; G_miny<=G_maxy; G_miny++, l_adr+=HW_SCREEN_X_SIZE)
   {
     adr=l_adr+(beg=G_start[G_miny]);         /* addr of the current line */
     end=G_end[G_miny]-beg+1;                 /* ends here */

     memset((char *) adr, colour, end);
   }
 }
}


void PSB_GouraudFill(Tri *Face, Object *obj, Light *light, Camera *cam)
// trace une face en Gouraud Shading
{
  static int xa,ya,xb,yb,xc,yc,xb_prime;
  static int ind_a,ind_b,ind_c;
  static int ia,ib,ic,ib_prime;
  static long xdeb,xfin,ideb;
  static long pente1,pente2,pente3;
  static long incI1,incI2,incI3,incIx;
  int tmp;
  register int i;
  char *ChunkyScreen=G_buffer, *cnk;
  VERTEX *Point;

  ind_a=Face->PtVertex[0];
  ind_b=Face->PtVertex[1];
  ind_c=Face->PtVertex[2];

  Point=&obj->Vertex[ind_a];
  xa=Point->ScreenVertex.x;
  ya=Point->ScreenVertex.y;

  Point=&obj->Vertex[ind_b];
  xb=Point->ScreenVertex.x;
  yb=Point->ScreenVertex.y;

  Point=&obj->Vertex[ind_c];
  xc=Point->ScreenVertex.x;
  yc=Point->ScreenVertex.y;

  ia=(int)240*IntensityPoint(obj,light,cam,&obj->Vertex[ind_a]);
  ib=(int)240*IntensityPoint(obj,light,cam,&obj->Vertex[ind_b]);
  ic=(int)240*IntensityPoint(obj,light,cam,&obj->Vertex[ind_c]);

  // tri des 3 points sur l'axe des y
  if(yb<ya)
  {
    tmp=ya; ya=yb; yb=tmp;
    tmp=xa; xa=xb; xb=tmp;
    tmp=ia; ia=ib; ib=tmp;
  }
  if(yc<ya)
  {
    tmp=ya; ya=yc; yc=tmp;
    tmp=xa; xa=xc; xc=tmp;
    tmp=ia; ia=ic; ic=tmp;
  }
  if(yc<yb)
  {
    tmp=yb; yb=yc; yc=tmp;
    tmp=xb; xb=xc; xc=tmp;
    tmp=ib; ib=ic; ic=tmp;
  }

  // calcul des pentes
  if(yb>ya) pente1=((xb-xa)<<16)/(yb-ya);
  if(yc>yb) pente2=((xc-xb)<<16)/(yc-yb);
  if(yc>ya) pente3=((xc-xa)<<16)/(yc-ya); else return;

  // calcul des increments d'intensite sur les pentes
  if(yb>ya) incI1=((ib-ia)<<16)/(yb-ya);
  if(yc>yb) incI2=((ic-ib)<<16)/(yc-yb);
  incI3=((ic-ia)<<16)/(yc-ya);

  // calcul du point xb'
  xb_prime=((pente3*(yb-ya))>>16)+xa;
  ib_prime=((incI3*(yb-ya))>>16)+ia;
  if(xb_prime==xb)
    incIx=(ib_prime-ib)<<16;
  else
    incIx=((ib_prime-ib)<<16)/(xb_prime-xb);

  // placement sur la ligne ya
  ChunkyScreen+=HW_SCREEN_X_SIZE*ya;


  if(ya<yb)
  if(pente1<pente3)
  {
    // dessine de gauche a droite

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, ideb=ia<<16; i<yb; i++)
    {
      register long j,icurr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr=ideb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=icurr>>16;
	icurr+=incIx;
      }
      xdeb+=pente1; xfin+=pente3;
      ideb+=incI1;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb<<16, xfin=xb_prime<<16; i<=yc; i++)
    {
      register long j,icurr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr=ideb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=icurr>>16;
	icurr+=incIx;
      }
      xdeb+=pente2; xfin+=pente3;
      ideb+=incI2;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    // dessine de droite a gauche

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, ideb=ia<<16; i<yb; i++)
    {
      register long j,icurr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr=ideb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=icurr>>16;
	icurr+=incIx;
      }
      xdeb+=pente3; xfin+=pente1;
      ideb+=incI3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb_prime<<16, xfin=xb<<16; i<=yc; i++)
    {
      register long j,icurr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr=ideb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=icurr>>16;
	icurr+=incIx;
      }
      xdeb+=pente3; xfin+=pente2;
      ideb+=incI3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    if(xa<xb)
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xa<<16, xfin=xb<<16, ideb=ia<<16; i<=yc; i++)
      {
	register long j,icurr;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	icurr=ideb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  *cnk++=icurr>>16;
	  icurr+=incIx;
	}
	xdeb+=pente3; xfin+=pente2;
	ideb+=incI3;
	ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
    else
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xb<<16, xfin=xa<<16, ideb=ib<<16; i<=yc; i++)
      {
	register long j,icurr;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	icurr=ideb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  *cnk++=icurr>>16;
	  icurr+=incIx;
	}
	xdeb+=pente2; xfin+=pente3;
	ideb+=incI2;
	ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
  }
}

void PSB_GouraudColorFill(Tri *Face, Object *obj, Light *light, Camera *cam)
// trace une face en Gouraud Shading
{
  static int xa,ya,xb,yb,xc,yc,xb_prime;
  static int ind_a,ind_b,ind_c;
  static int ia_r,ib_r,ic_r,ib_prime_r;
  static int ia_g,ib_g,ic_g,ib_prime_g;
  static int ia_b,ib_b,ic_b,ib_prime_b;
  static long xdeb,xfin,ideb_r,ideb_g,ideb_b;
  static long pente1,pente2,pente3;
  static long incIr1,incIr2,incIr3,incIrx;
  static long incIg1,incIg2,incIg3,incIgx;
  static long incIb1,incIb2,incIb3,incIbx;
  Color col_a,col_b,col_c;
  long icurr_r,icurr_g,icurr_b;
  int tmp;
  register int i;
  char *ChunkyScreen=G_buffer, *cnk;
  VERTEX *Point;

  ind_a=Face->PtVertex[0];
  ind_b=Face->PtVertex[1];
  ind_c=Face->PtVertex[2];

  Point=&obj->Vertex[ind_a];
  xa=Point->ScreenVertex.x;
  ya=Point->ScreenVertex.y;

  Point=&obj->Vertex[ind_b];
  xb=Point->ScreenVertex.x;
  yb=Point->ScreenVertex.y;

  Point=&obj->Vertex[ind_c];
  xc=Point->ScreenVertex.x;
  yc=Point->ScreenVertex.y;

  // calcule des couleurs aux 3 sommets
  col_a=RGBIntensityPoint(obj,light,cam,&obj->Vertex[ind_a]);
  col_b=RGBIntensityPoint(obj,light,cam,&obj->Vertex[ind_b]);
  col_c=RGBIntensityPoint(obj,light,cam,&obj->Vertex[ind_c]);
  ia_r=col_a.r; ia_g=col_a.g; ia_b=col_a.b;
  ib_r=col_b.r; ib_g=col_b.g; ib_b=col_b.b;
  ic_r=col_c.r; ic_g=col_c.g; ic_b=col_c.b;

  // tri des 3 points sur l'axe des y
  if(yb<ya)
  {
    tmp=xa; xa=xb; xb=tmp;
    tmp=ya; ya=yb; yb=tmp;
    tmp=ia_r; ia_r=ib_r; ib_r=tmp;
    tmp=ia_g; ia_g=ib_g; ib_g=tmp;
    tmp=ia_b; ia_b=ib_b; ib_b=tmp;
  }
  if(yc<ya)
  {
    tmp=xa; xa=xc; xc=tmp;
    tmp=ya; ya=yc; yc=tmp;
    tmp=ia_r; ia_r=ic_r; ic_r=tmp;
    tmp=ia_g; ia_g=ic_g; ic_g=tmp;
    tmp=ia_b; ia_b=ic_b; ic_b=tmp;
  }
  if(yc<yb)
  {
    tmp=xb; xb=xc; xc=tmp;
    tmp=yb; yb=yc; yc=tmp;
    tmp=ib_r; ib_r=ic_r; ic_r=tmp;
    tmp=ib_g; ib_g=ic_g; ic_g=tmp;
    tmp=ib_b; ib_b=ic_b; ic_b=tmp;
  }

  // calcul des pentes
  if(yb>ya) pente1=((xb-xa)<<16)/(yb-ya);
  if(yc>yb) pente2=((xc-xb)<<16)/(yc-yb);
  if(yc>ya) pente3=((xc-xa)<<16)/(yc-ya); else return;

  // calcul des increments d'intensite sur les pentes
  if(yb>ya)
  {
    incIr1=((ib_r-ia_r)<<16)/(yb-ya);
    incIg1=((ib_g-ia_g)<<16)/(yb-ya);
    incIb1=((ib_b-ia_b)<<16)/(yb-ya);
  }
  if(yc>yb)
  {
    incIr2=((ic_r-ib_r)<<16)/(yc-yb);
    incIg2=((ic_g-ib_g)<<16)/(yc-yb);
    incIb2=((ic_b-ib_b)<<16)/(yc-yb);
  }
  incIr3=((ic_r-ia_r)<<16)/(yc-ya);
  incIg3=((ic_g-ia_g)<<16)/(yc-ya);
  incIb3=((ic_b-ia_b)<<16)/(yc-ya);

   // calcul du point xb'
  xb_prime=((pente3*(yb-ya))>>16)+xa;
  ib_prime_r=((incIr3*(yb-ya))>>16)+ia_r;
  ib_prime_g=((incIg3*(yb-ya))>>16)+ia_g;
  ib_prime_b=((incIb3*(yb-ya))>>16)+ia_b;
  if(xb_prime==xb)
  {
    incIrx=(ib_prime_r-ib_r)<<16;
    incIgx=(ib_prime_g-ib_g)<<16;
    incIbx=(ib_prime_b-ib_b)<<16;
  }
  else
  {
    incIrx=((ib_prime_r-ib_r)<<16)/(xb_prime-xb);
    incIgx=((ib_prime_g-ib_g)<<16)/(xb_prime-xb);
    incIbx=((ib_prime_b-ib_b)<<16)/(xb_prime-xb);
  }

  // placement sur la ligne ya
  ChunkyScreen+=HW_SCREEN_X_SIZE*ya;


  if(ya<yb)
  if(pente1<pente3)
  {
    // dessine de gauche a droite

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, ideb_r=ia_r<<16, ideb_g=ia_g<<16, ideb_b=ia_b<<16; i<yb; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr_r=ideb_r; icurr_g=ideb_g; icurr_b=ideb_b;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=CouleurTramee(j,i,icurr_r>>16,icurr_g>>16,icurr_b>>16);
	icurr_r+=incIrx; icurr_g+=incIgx; icurr_b+=incIbx;
      }
      xdeb+=pente1; xfin+=pente3;
      ideb_r+=incIr1; ideb_g+=incIg1; ideb_b+=incIb1;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb<<16, xfin=xb_prime<<16; i<=yc; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr_r=ideb_r; icurr_g=ideb_g; icurr_b=ideb_b;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=CouleurTramee(j,i,icurr_r>>16,icurr_g>>16,icurr_b>>16);
	icurr_r+=incIrx; icurr_g+=incIgx; icurr_b+=incIbx;
      }
      xdeb+=pente2; xfin+=pente3;
      ideb_r+=incIr2; ideb_g+=incIg2; ideb_b+=incIb2;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    // dessine de droite a gauche

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, ideb_r=ia_r<<16, ideb_g=ia_g<<16, ideb_b=ia_b<<16; i<yb; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr_r=ideb_r; icurr_g=ideb_g; icurr_b=ideb_b;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=CouleurTramee(j,i,icurr_r>>16,icurr_g>>16,icurr_b>>16);
	icurr_r+=incIrx; icurr_g+=incIgx; icurr_b+=incIbx;
      }
      xdeb+=pente3; xfin+=pente1;
      ideb_r+=incIr3; ideb_g+=incIg3; ideb_b+=incIb3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb_prime<<16, xfin=xb<<16; i<=yc; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      icurr_r=ideb_r; icurr_g=ideb_g; icurr_b=ideb_b;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=CouleurTramee(j,i,icurr_r>>16,icurr_g>>16,icurr_b>>16);
	icurr_r+=incIrx; icurr_g+=incIgx; icurr_b+=incIbx;
      }
      xdeb+=pente3; xfin+=pente2;
      ideb_r+=incIr3; ideb_g+=incIg3; ideb_b+=incIb3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    if(xa<xb)
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xa<<16, xfin=xb<<16, ideb_r=ia_r<<16, ideb_g=ia_g<<16, ideb_b=ia_b<<16; i<=yc; i++)
      {
	register long j;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	icurr_r=ideb_r; icurr_g=ideb_g; icurr_b=ideb_b;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  *cnk++=CouleurTramee(j,i,icurr_r>>16,icurr_g>>16,icurr_b>>16);
	  icurr_r+=incIrx; icurr_g+=incIgx; icurr_b+=incIbx;
	}
	xdeb+=pente3; xfin+=pente2;
	ideb_r+=incIr3; ideb_g+=incIg3; ideb_b+=incIb3;
	ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
    else
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xb<<16, xfin=xa<<16, ideb_r=ib_r<<16, ideb_g=ib_g<<16, ideb_b=ib_b<<16; i<=yc; i++)
      {
	register long j;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	icurr_r=ideb_r; icurr_g=ideb_g; icurr_b=ideb_b;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  *cnk++=CouleurTramee(j,i,icurr_r>>16,icurr_g>>16,icurr_b>>16);
	  icurr_r+=incIrx; icurr_g+=incIgx; icurr_b+=incIbx;
	}
	xdeb+=pente2; xfin+=pente3;
	ideb_r+=incIr2; ideb_g+=incIg2; ideb_b+=incIb2;
	ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
  }
}

void PSB_GouraudFill_1Line(int xa, int xb, int xc, int ia, int ib, int ic, char *cnk)
{
  register int i;
  register long int icurr,incIx;

  if(xa<xb)
  {
    if(xa<xc)
    {
      // xa est le min des x
      cnk+=xa;
      if(xb>xc)
      {
	// xc est le max des x
	if(xc==xa)
	{
	  *cnk=ia;
	  return;
	}
	icurr=ia<<16;
	incIx=((ic-ia)<<16)/(xc-xa);
	for(i=xa; i<=xc; i++)
	{
	  *cnk++=icurr>>16;
	  icurr+=incIx;
	}
      }
      else
      {
	// xb est le max des x
	if(xb==xa)
	{
	  *cnk=ia;
	  return;
	}
	icurr=ia<<16;
	incIx=((ib-ia)<<16)/(xb-xa);
	for(i=xa; i<=xb; i++)
	{
	  *cnk++=icurr>>16;
	  icurr+=incIx;
	}
      }
    }
    else
    {
      if(xb<xc)
      {
	// xb est le min des x
	cnk+=xb;
	if(xa>xc)
	{
	  // xa est le max des x
	  if(xa==xb)
	  {
	    *cnk=ia;
	    return;
	  }
	  icurr=ib<<16;
	  incIx=((ia-ib)<<16)/(xa-xb);
	  for(i=xb; i<=xa; i++)
	  {
	    *cnk++=icurr>>16;
	    icurr+=incIx;
	  }
	}
	else
        {
	  // xc est le max des x
	  if(xc==xb)
	  {
	    *cnk=ia;
	    return;
	  }
	  icurr=ib<<16;
	  incIx=((ic-ib)<<16)/(xc-xb);
	  for(i=xb; i<=xc; i++)
	  {
	    *cnk++=icurr>>16;
	    icurr+=incIx;
	  }
	}
      }
      else
      {
	// xc est le min des x
	cnk+=xc;
	if(xa>xb)
	{
	  // xa est le max des x
	  if(xa==xc)
	  {
	    *cnk=ia;
	    return;
	  }
	  icurr=ic<<16;
	  incIx=((ia-ic)<<16)/(xa-xc);
	  for(i=xc; i<=xa; i++)
	  {
	    *cnk++=icurr>>16;
	    icurr+=incIx;
	  }
	}
	else
	{
	  // xb est le max des x
	  if(xb==xc)
	  {
	    *cnk=ia;
	    return;
	  }
	  icurr=ic<<16;
	  incIx=((ib-ic)<<16)/(xb-xc);
	  for(i=xc; i<=xb; i++)
	  {
	    *cnk++=icurr>>16;
	    icurr+=incIx;
	  }
	}
      }
    }
  }
}


void PSB_PhongFill(Tri *Face, Object *obj, Light *light, Camera *cam)
// trace une face en Gouraud Shading
{
  static int xa,ya,xb,yb,xc,yc,xb_prime;
  static int ind_a,ind_b,ind_c;
  static long xdeb,xfin;
  static long pente1,pente2,pente3;
  static double incNx1,incNy1,incNz1,incNx2,incNy2,incNz2,incNx3,incNy3,incNz3,incNxx,incNyx,incNzx;
  static double incPx1,incPy1,incPz1,incPx2,incPy2,incPz2,incPx3,incPy3,incPz3,incPxx,incPyx,incPzx;
  int tmp;
  register int i;
  char *ChunkyScreen=G_buffer, *cnk;
  VERTEX *Point;
  VERTEX Na,Nb,Nc,Nb_prime,Ntmp,Ndeb,Ncurr;

  ind_a=Face->PtVertex[0];
  ind_b=Face->PtVertex[1];
  ind_c=Face->PtVertex[2];

  Point=&obj->Vertex[ind_a];
  xa=Point->ScreenVertex.x;
  ya=Point->ScreenVertex.y;
  Na.CameraVertex=Point->CameraVertex;
  Na.N=Point->N;

  Point=&obj->Vertex[ind_b];
  xb=Point->ScreenVertex.x;
  yb=Point->ScreenVertex.y;
  Nb.CameraVertex=Point->CameraVertex;
  Nb.N=Point->N;

  Point=&obj->Vertex[ind_c];
  xc=Point->ScreenVertex.x;
  yc=Point->ScreenVertex.y;
  Nc.CameraVertex=Point->CameraVertex;
  Nc.N=Point->N;

  // tri des 3 points sur l'axe des y
  if(yb<ya)
  {
    tmp=xa; xa=xb; xb=tmp;
    tmp=ya; ya=yb; yb=tmp;
    Ntmp=Na; Na=Nb; Nb=Ntmp;
  }
  if(yc<ya)
  {
    tmp=xa; xa=xc; xc=tmp;
    tmp=ya; ya=yc; yc=tmp;
    Ntmp=Na; Na=Nc; Nc=Ntmp;
  }
  if(yc<yb)
  {
    tmp=xb; xb=xc; xc=tmp;
    tmp=yb; yb=yc; yc=tmp;
    Ntmp=Nb; Nb=Nc; Nc=Ntmp;
  }

  // calcul des pentes
  if(yb>ya) pente1=((xb-xa)<<16)/(yb-ya);
  if(yc>yb) pente2=((xc-xb)<<16)/(yc-yb);
  if(yc>ya) pente3=((xc-xa)<<16)/(yc-ya); else return;

  // calcul des increments de normales sur les pentes
  if(yb>ya)
  {
    incNx1=(double)(Nb.N.x-Na.N.x)/(double)(yb-ya);
    incNy1=(double)(Nb.N.y-Na.N.y)/(double)(yb-ya);
    incNz1=(double)(Nb.N.z-Na.N.z)/(double)(yb-ya);
    incPx1=(double)(Nb.CameraVertex.x-Na.CameraVertex.x)/(double)(yb-ya);
    incPy1=(double)(Nb.CameraVertex.y-Na.CameraVertex.y)/(double)(yb-ya);
    incPz1=(double)(Nb.CameraVertex.z-Na.CameraVertex.z)/(double)(yb-ya);
  }
  if(yc>yb)
  {
    incNx2=(double)(Nc.N.x-Nb.N.x)/(double)(yc-yb);
    incNy2=(double)(Nc.N.y-Nb.N.y)/(double)(yc-yb);
    incNz2=(double)(Nc.N.z-Nb.N.z)/(double)(yc-yb);
    incPx2=(double)(Nc.CameraVertex.x-Nb.CameraVertex.x)/(double)(yc-yb);
    incPy2=(double)(Nc.CameraVertex.y-Nb.CameraVertex.y)/(double)(yc-yb);
    incPz2=(double)(Nc.CameraVertex.z-Nb.CameraVertex.z)/(double)(yc-yb);
  }
  incNx3=(double)(Nc.N.x-Na.N.x)/(double)(yc-ya);
  incNy3=(double)(Nc.N.y-Na.N.y)/(double)(yc-ya);
  incNz3=(double)(Nc.N.z-Na.N.z)/(double)(yc-ya);
  incPx3=(double)(Nc.CameraVertex.x-Na.CameraVertex.x)/(double)(yc-ya);
  incPy3=(double)(Nc.CameraVertex.y-Na.CameraVertex.y)/(double)(yc-ya);
  incPz3=(double)(Nc.CameraVertex.z-Na.CameraVertex.z)/(double)(yc-ya);

  // calcul du point xb'
  xb_prime=((pente3*(yb-ya))>>16)+xa;
  Nb_prime.N.x=incNx3*(double)(yb-ya)+Na.N.x;
  Nb_prime.N.y=incNy3*(double)(yb-ya)+Na.N.y;
  Nb_prime.N.z=incNz3*(double)(yb-ya)+Na.N.z;
  Nb_prime.CameraVertex.x=incPx3*(double)(yb-ya)+Na.CameraVertex.x;
  Nb_prime.CameraVertex.y=incPy3*(double)(yb-ya)+Na.CameraVertex.y;
  Nb_prime.CameraVertex.z=incPz3*(double)(yb-ya)+Na.CameraVertex.z;
  if(xb_prime==xb)
  {
    incNxx=Nb_prime.N.x-Nb.N.x;
    incNyx=Nb_prime.N.y-Nb.N.y;
    incNzx=Nb_prime.N.z-Nb.N.z;
    incPxx=Nb_prime.CameraVertex.x-Nb.CameraVertex.x;
    incPyx=Nb_prime.CameraVertex.y-Nb.CameraVertex.y;
    incPzx=Nb_prime.CameraVertex.z-Nb.CameraVertex.z;
  }
  else
  {
    incNxx=(Nb_prime.N.x-Nb.N.x)/(double)(xb_prime-xb);
    incNyx=(Nb_prime.N.y-Nb.N.y)/(double)(xb_prime-xb);
    incNzx=(Nb_prime.N.z-Nb.N.z)/(double)(xb_prime-xb);
    incPxx=(Nb_prime.CameraVertex.x-Nb.CameraVertex.x)/(double)(xb_prime-xb);
    incPyx=(Nb_prime.CameraVertex.y-Nb.CameraVertex.y)/(double)(xb_prime-xb);
    incPzx=(Nb_prime.CameraVertex.z-Nb.CameraVertex.z)/(double)(xb_prime-xb);
  }

  // placement sur la ligne ya
  ChunkyScreen+=HW_SCREEN_X_SIZE*ya;


  if(ya<yb)
  if(pente1<pente3)
  {
    // dessine de gauche a droite

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, Ndeb=Na; i<yb; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      Ncurr=Ndeb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=(int)(240*IntensityPoint(obj,light,cam,&Ncurr));
	Ncurr.N.x+=incNxx; Ncurr.N.y+=incNyx; Ncurr.N.z+=incNzx;
	Ncurr.CameraVertex.x+=incPxx; Ncurr.CameraVertex.y+=incPyx; Ncurr.CameraVertex.z+=incPzx;
      }
      xdeb+=pente1; xfin+=pente3;
      Ndeb.N.x+=incNx1; Ndeb.N.y+=incNy1; Ndeb.N.z+=incNz1;
      Ndeb.CameraVertex.x+=incPx1; Ndeb.CameraVertex.y+=incPy1; Ndeb.CameraVertex.z+=incPz1;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb<<16, xfin=xb_prime<<16; i<=yc; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      Ncurr=Ndeb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=(int)(240*IntensityPoint(obj,light,cam,&Ncurr));
	Ncurr.N.x+=incNxx; Ncurr.N.y+=incNyx; Ncurr.N.z+=incNzx;
	Ncurr.CameraVertex.x+=incPxx; Ncurr.CameraVertex.y+=incPyx; Ncurr.CameraVertex.z+=incPzx;
      }
      xdeb+=pente2; xfin+=pente3;
      Ndeb.N.x+=incNx2; Ndeb.N.y+=incNy2; Ndeb.N.z+=incNz2;
      Ndeb.CameraVertex.x+=incPx2; Ndeb.CameraVertex.y+=incPy2; Ndeb.CameraVertex.z+=incPz2;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    // dessine de droite a gauche

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, Ndeb=Na; i<yb; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      Ncurr=Ndeb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=(int)(240*IntensityPoint(obj,light,cam,&Ncurr));
	Ncurr.N.x+=incNxx; Ncurr.N.y+=incNyx; Ncurr.N.z+=incNzx;
	Ncurr.CameraVertex.x+=incPxx; Ncurr.CameraVertex.y+=incPyx; Ncurr.CameraVertex.z+=incPzx;
      }
      xdeb+=pente3; xfin+=pente1;
      Ndeb.N.x+=incNx3; Ndeb.N.y+=incNy3; Ndeb.N.z+=incNz3;
      Ndeb.CameraVertex.x+=incPx3; Ndeb.CameraVertex.y+=incPy3; Ndeb.CameraVertex.z+=incPz3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb_prime<<16, xfin=xb<<16; i<=yc; i++)
    {
      register long j;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      Ncurr=Ndeb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	*cnk++=(int)(240*IntensityPoint(obj,light,cam,&Ncurr));
	Ncurr.N.x+=incNxx; Ncurr.N.y+=incNyx; Ncurr.N.z+=incNzx;
	Ncurr.CameraVertex.x+=incPxx; Ncurr.CameraVertex.y+=incPyx; Ncurr.CameraVertex.z+=incPzx;
      }
      xdeb+=pente3; xfin+=pente2;
      Ndeb.N.x+=incNx3; Ndeb.N.y+=incNy3; Ndeb.N.z+=incNz3;
      Ndeb.CameraVertex.x+=incPx3; Ndeb.CameraVertex.y+=incPy3; Ndeb.CameraVertex.z+=incPz3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    } 
  }
  else
  {
    if(xa<xb)
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xa<<16, xfin=xb<<16, Ndeb=Nb; i<=yc; i++)
      {
	register long j;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	Ncurr=Ndeb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  *cnk++=(int)(240*IntensityPoint(obj,light,cam,&Ncurr));
	  Ncurr.N.x+=incNxx; Ncurr.N.y+=incNyx; Ncurr.N.z+=incNzx;
	  Ncurr.CameraVertex.x+=incPxx; Ncurr.CameraVertex.y+=incPyx; Ncurr.CameraVertex.z+=incPzx;
	}
	xdeb+=pente3; xfin+=pente2;
	Ndeb.N.x+=incNx3; Ndeb.N.y+=incNy3; Ndeb.N.z+=incNz3;
        Ndeb.CameraVertex.x+=incPx3; Ndeb.CameraVertex.y+=incPy3; Ndeb.CameraVertex.z+=incPz3;
	ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
    else
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xb<<16, xfin=xa<<16, Ndeb=Na; i<=yc; i++)
      {
        register long j;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	Ncurr=Ndeb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  *cnk++=(int)(240*IntensityPoint(obj,light,cam,&Ncurr));
          Ncurr.N.x+=incNxx; Ncurr.N.y+=incNyx; Ncurr.N.z+=incNzx;
          Ncurr.CameraVertex.x+=incPxx; Ncurr.CameraVertex.y+=incPyx; Ncurr.CameraVertex.z+=incPzx;
        }
        xdeb+=pente2; xfin+=pente3;
        Ndeb.N.x+=incNx2; Ndeb.N.y+=incNy2; Ndeb.N.z+=incNz2;
        Ndeb.CameraVertex.x+=incPx2; Ndeb.CameraVertex.y+=incPy2; Ndeb.CameraVertex.z+=incPz2;
        ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
  }
}


void PSB_EnvFill(Tri *Face, Object *obj, Camera *cam, char *Texture)
// trace une face en Environnement Mapping
{
  static int xa,ya,xb,yb,xc,yc,xb_prime;
  static int ind_a,ind_b,ind_c;
  static int xa_map,xb_map,xc_map,xb_prime_map;
  static int ya_map,yb_map,yc_map,yb_prime_map;
  static long xdeb,xfin,xmap_deb,ymap_deb;
  static long pente1,pente2,pente3;
  static long incX1,incX2,incX3,incXx;
  static long incY1,incY2,incY3,incYx;
  int tmp;
  register int i;
  char *ChunkyScreen=G_buffer, *cnk;
  VERTEX *Point;
  COORD3D N;

  ind_a=Face->PtVertex[0];
  ind_b=Face->PtVertex[1];
  ind_c=Face->PtVertex[2];

  Point=&obj->Vertex[ind_a];
  xa=Point->ScreenVertex.x;
  ya=Point->ScreenVertex.y;
  TransformVector(&Point->N,&obj->LocalToCamera,&N);
  xa_map=(int)(100*N.x)+128;
  ya_map=(int)(100*N.y)+128;

  Point=&obj->Vertex[ind_b];
  xb=Point->ScreenVertex.x;
  yb=Point->ScreenVertex.y;
  TransformVector(&Point->N,&obj->LocalToCamera,&N);
  xb_map=(int)(100*N.x)+128;
  yb_map=(int)(100*N.y)+128;

  Point=&obj->Vertex[ind_c];
  xc=Point->ScreenVertex.x;
  yc=Point->ScreenVertex.y;
  TransformVector(&Point->N,&obj->LocalToCamera,&N);
  xc_map=(int)(100*N.x)+128;
  yc_map=(int)(100*N.y)+128;


  // tri des 3 points sur l'axe des y
  if(yb<ya)
  {
    tmp=ya; ya=yb; yb=tmp;
    tmp=xa; xa=xb; xb=tmp;
    tmp=xa_map; xa_map=xb_map; xb_map=tmp;
    tmp=ya_map; ya_map=yb_map; yb_map=tmp;
  }
  if(yc<ya)
  {
    tmp=ya; ya=yc; yc=tmp;
    tmp=xa; xa=xc; xc=tmp;
    tmp=xa_map; xa_map=xc_map; xc_map=tmp;
    tmp=ya_map; ya_map=yc_map; yc_map=tmp;
  }
  if(yc<yb)
  {
    tmp=yb; yb=yc; yc=tmp;
    tmp=xb; xb=xc; xc=tmp;
    tmp=xb_map; xb_map=xc_map; xc_map=tmp;
    tmp=yb_map; yb_map=yc_map; yc_map=tmp;
  }


  // calcul des pentes
  if(yb>ya)
    pente1=((xb-xa)<<16)/(yb-ya);
  else
    pente1=0;
  if(yc>yb)
    pente2=((xc-xb)<<16)/(yc-yb);
  else
    pente2=0;
  if(yc>ya)
    pente3=((xc-xa)<<16)/(yc-ya);
  else
  {
    // la face est reduite a une ligne horizontale
    return;
  }

  // calcul des increments mappings sur les pentes
  if(yb>ya)
  {
    incX1=((xb_map-xa_map)<<8)/(yb-ya);
    incY1=((yb_map-ya_map)<<8)/(yb-ya);
  }
  else
  {
    incX1=0;
    incY1=0;
  }
  if(yc>yb)
  {
    incX2=((xc_map-xb_map)<<8)/(yc-yb);
    incY2=((yc_map-yb_map)<<8)/(yc-yb);
  }
  else
  {
    incX2=0;
    incY2=0;
  }
  incX3=((xc_map-xa_map)<<8)/(yc-ya);
  incY3=((yc_map-ya_map)<<8)/(yc-ya);


  // calcul du point xb'
  xb_prime=((pente3*(yb-ya))>>16)+xa;
  xb_prime_map=((incX3*(yb-ya))>>8)+xa_map;
  yb_prime_map=((incY3*(yb-ya))>>8)+ya_map;
  if(xb_prime==xb)
  {
    incXx=(xb_prime_map-xb_map)<<8;
    incYx=(yb_prime_map-yb_map)<<8;
  }
  else
  {
    incXx=((xb_prime_map-xb_map)<<8)/(xb_prime-xb);
    incYx=((yb_prime_map-yb_map)<<8)/(xb_prime-xb);
  }

  // placement sur la ligne ya
  ChunkyScreen+=HW_SCREEN_X_SIZE*ya;


  if(ya<yb)
  if(pente1<pente3)
  {
    // dessine de gauche a droite

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, xmap_deb=xa_map<<8, ymap_deb=ya_map<<8; i<yb; i++)
    {
      register long j,xmap_curr,ymap_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	  *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	xmap_curr+=incXx; ymap_curr+=incYx;
      }
      xdeb+=pente1; xfin+=pente3;
      xmap_deb+=incX1; ymap_deb+=incY1;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb<<16, xfin=xb_prime<<16; i<=yc; i++)
    {
      register long j,xmap_curr,ymap_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	  *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	xmap_curr+=incXx; ymap_curr+=incYx;
      }
      xdeb+=pente2; xfin+=pente3;
      xmap_deb+=incX2; ymap_deb+=incY2;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    // dessine de droite a gauche

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, xmap_deb=xa_map<<8, ymap_deb=ya_map<<8; i<yb; i++)
    {
      register long j,xmap_curr,ymap_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	  *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	xmap_curr+=incXx; ymap_curr+=incYx;
      }
      xdeb+=pente3; xfin+=pente1;
      xmap_deb+=incX3; ymap_deb+=incY3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb_prime<<16, xfin=xb<<16; i<=yc; i++)
    {
      register long j,xmap_curr,ymap_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	  *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	xmap_curr+=incXx; ymap_curr+=incYx;
      }
      xdeb+=pente3; xfin+=pente2;
      xmap_deb+=incX3; ymap_deb+=incY3;
      ChunkyScreen+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    if(xa<xb)
    {
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xa<<16, xfin=xb<<16, xmap_deb=xa_map<<8, ymap_deb=ya_map<<8; i<=yc; i++)
      {
	register long j,xmap_curr,ymap_curr;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	xmap_curr=xmap_deb; ymap_curr=ymap_deb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	    *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	  xmap_curr+=incXx; ymap_curr+=incYx;
	}
	xdeb+=pente3; xfin+=pente2;
	xmap_deb+=incX3; ymap_deb+=incY3;
	ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
    }
    else
    {
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xb<<16, xfin=xa<<16, xmap_deb=xb_map<<8, ymap_deb=yb_map<<8; i<=yc; i++)
      {
	register long j,xmap_curr,ymap_curr;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16);
	xmap_curr=xmap_deb; ymap_curr=ymap_deb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	    *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	  xmap_curr+=incXx; ymap_curr+=incYx;
	}
	xdeb+=pente2; xfin+=pente3;
	xmap_deb+=incX2; ymap_deb+=incY2;
	ChunkyScreen+=HW_SCREEN_X_SIZE;
      }
    }
  }
}


void PSB_EnvZBufferFill(Tri *Face, Object *obj, Camera *cam, char *Texture)
// trace une face en Environnement Mapping avec Z-Buffer
{
  static int xa,ya,xb,yb,xc,yc,xb_prime;
  static int ind_a,ind_b,ind_c;
  static int xa_map,xb_map,xc_map,xb_prime_map;
  static int ya_map,yb_map,yc_map,yb_prime_map;
  static double za,zb,zc,zb_prime;
  static long xdeb,xfin,xmap_deb,ymap_deb,z_deb;
  static long pente1,pente2,pente3;
  static long incX1,incX2,incX3,incXx;
  static long incY1,incY2,incY3,incYx;
  static double incZ1,incZ2,incZ3,incZx;
  long tmp;
  double tmp2;
  register int i;
  char *ChunkyScreen=G_buffer, *cnk;
  double *ZBufferLine=cam->zbuffer, *zbuf;
  VERTEX *Point;
  COORD3D N;

  ind_a=Face->PtVertex[0];
  ind_b=Face->PtVertex[1];
  ind_c=Face->PtVertex[2];

  Point=&obj->Vertex[ind_a];
  xa=Point->ScreenVertex.x;
  ya=Point->ScreenVertex.y;
  TransformVector(&Point->N,&obj->LocalToCamera,&N);
  xa_map=(int)(100*N.x)+128;
  ya_map=(int)(100*N.y)+128;
  za=1/Point->CameraVertex.z;

  Point=&obj->Vertex[ind_b];
  xb=Point->ScreenVertex.x;
  yb=Point->ScreenVertex.y;
  TransformVector(&Point->N,&obj->LocalToCamera,&N);
  xb_map=(int)(100*N.x)+128;
  yb_map=(int)(100*N.y)+128;
  zb=1/Point->CameraVertex.z;

  Point=&obj->Vertex[ind_c];
  xc=Point->ScreenVertex.x;
  yc=Point->ScreenVertex.y;
  TransformVector(&Point->N,&obj->LocalToCamera,&N);
  xc_map=(int)(100*N.x)+128;
  yc_map=(int)(100*N.y)+128;
  zc=1/Point->CameraVertex.z;


  // tri des 3 points sur l'axe des y
  if(yb<ya)
  {
    tmp=xa; xa=xb; xb=tmp;
    tmp=ya; ya=yb; yb=tmp;
    tmp2=za; za=zb; zb=tmp2;
    tmp=xa_map; xa_map=xb_map; xb_map=tmp;
    tmp=ya_map; ya_map=yb_map; yb_map=tmp;    
  }
  if(yc<ya)
  {
    tmp=xa; xa=xc; xc=tmp;
    tmp=ya; ya=yc; yc=tmp;
    tmp2=za; za=zc; zc=tmp2;
    tmp=xa_map; xa_map=xc_map; xc_map=tmp;
    tmp=ya_map; ya_map=yc_map; yc_map=tmp;
  }
  if(yc<yb)
  {
    tmp=xb; xb=xc; xc=tmp;
    tmp=yb; yb=yc; yc=tmp;
    tmp2=zb; zb=zc; zc=tmp2;
    tmp=xb_map; xb_map=xc_map; xc_map=tmp;
    tmp=yb_map; yb_map=yc_map; yc_map=tmp;
  }


  // calcul des pentes
  if(yb>ya) pente1=((xb-xa)<<16)/(yb-ya);
  if(yc>yb) pente2=((xc-xb)<<16)/(yc-yb);
  if(yc>ya) pente3=((xc-xa)<<16)/(yc-ya); else return;


  // calcul des increments mappings et des increments en z sur les pentes
  if(yb>ya)
  {
    incX1=((xb_map-xa_map)<<8)/(yb-ya);
    incY1=((yb_map-ya_map)<<8)/(yb-ya);
    incZ1=(zb-za)/(yb-ya);
  }
  if(yc>yb)
  {
    incX2=((xc_map-xb_map)<<8)/(yc-yb);
    incY2=((yc_map-yb_map)<<8)/(yc-yb);
    incZ2=(zc-zb)/(yc-yb);
  }
  incX3=((xc_map-xa_map)<<8)/(yc-ya);
  incY3=((yc_map-ya_map)<<8)/(yc-ya);
  incZ3=(zc-za)/(yc-ya);


  // calcul du point xb'
  xb_prime=((pente3*(yb-ya))>>16)+xa;
  xb_prime_map=((incX3*(yb-ya))>>8)+xa_map;
  yb_prime_map=((incY3*(yb-ya))>>8)+ya_map;
  zb_prime=(incZ3*(yb-ya))+za;
  if(xb_prime==xb)
  {
    incXx=(xb_prime_map-xb_map)<<8;
    incYx=(yb_prime_map-yb_map)<<8;
    incZx=(zb_prime-zb);
  }
  else
  {
    incXx=((xb_prime_map-xb_map)<<8)/(xb_prime-xb);
    incYx=((yb_prime_map-yb_map)<<8)/(xb_prime-xb);
    incZx=(zb_prime-zb)/(xb_prime-xb);
  }

  // placement sur la ligne ya
  ChunkyScreen+=HW_SCREEN_X_SIZE*ya;
  ZBufferLine +=HW_SCREEN_X_SIZE*ya;


  if(ya<yb)
  if(pente1<pente3)
  {
    // dessine de gauche a droite

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, xmap_deb=xa_map<<8, ymap_deb=ya_map<<8, z_deb=za; i<yb; i++)
    {
      register long j,xmap_curr,ymap_curr;
      register double z_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16); zbuf=ZBufferLine+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb; z_curr=z_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
{
  //printf("zbuffer=%ld\tz_curr=%ld\n",*zbuf,z_curr);
	  if(*zbuf++ > z_curr)
	  {
	    *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	    *(zbuf-1)=z_curr;
	  }
}
	xmap_curr+=incXx; ymap_curr+=incYx; z_curr+=incZx;
      }
      xdeb+=pente1; xfin+=pente3;
      xmap_deb+=incX1; ymap_deb+=incY1, z_deb+=incZ1;
      ChunkyScreen+=HW_SCREEN_X_SIZE; ZBufferLine+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb<<16, xfin=xb_prime<<16; i<=yc; i++)
    {
      register long j,xmap_curr,ymap_curr;
      register double z_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16); zbuf=ZBufferLine+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb; z_curr=z_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	  if(*zbuf++ > z_curr)
	  {
	    *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	    *(zbuf-1)=z_curr;
	  }
	xmap_curr+=incXx; ymap_curr+=incYx; z_curr+=incZx;
      }
      xdeb+=pente2; xfin+=pente3;
      xmap_deb+=incX2; ymap_deb+=incY2; z_deb+=incZ2;
      ChunkyScreen+=HW_SCREEN_X_SIZE; ZBufferLine+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    // dessine de droite a gauche

    // trace la 1ere partie du triangle
    for(i=ya, xdeb=xfin=xa<<16, xmap_deb=xa_map<<8, ymap_deb=ya_map<<8, z_deb=za; i<yb; i++)
    {
      register long j,xmap_curr,ymap_curr;
      register double z_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16); zbuf=ZBufferLine+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb; z_curr=z_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
{
	  if(*zbuf++ > z_curr)
	  {
	    *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	    *(zbuf-1)=z_curr;
	  }
}
	xmap_curr+=incXx; ymap_curr+=incYx; z_curr+=incZx;
      }
      xdeb+=pente3; xfin+=pente1;
      xmap_deb+=incX3; ymap_deb+=incY3; z_deb+=incZ3;
      ChunkyScreen+=HW_SCREEN_X_SIZE; ZBufferLine+=HW_SCREEN_X_SIZE;
    }

    // trace la 2nde partie du triangle
    for(i=yb, xdeb=xb_prime<<16, xfin=xb<<16; i<=yc; i++)
    {
      register long j,xmap_curr,ymap_curr;
      register double z_curr;
      if(i>=HW_SCREEN_Y_SIZE) return;
      cnk=ChunkyScreen+(xdeb>>16); zbuf=ZBufferLine+(xdeb>>16);
      xmap_curr=xmap_deb; ymap_curr=ymap_deb; z_curr=z_deb;
      for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
      {
	if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	  if(*zbuf++ > z_curr)
	  {
	    *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	    *(zbuf-1)=z_curr;
	  }
       	xmap_curr+=incXx; ymap_curr+=incYx; z_curr+=incZx;
      }
      xdeb+=pente3; xfin+=pente2;
      xmap_deb+=incX3; ymap_deb+=incY3; z_deb+=incZ3;
      ChunkyScreen+=HW_SCREEN_X_SIZE; ZBufferLine+=HW_SCREEN_X_SIZE;
    }
  }
  else
  {
    if(xa<xb)
    {
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xa<<16, xfin=xb<<16, xmap_deb=xa_map<<8, ymap_deb=ya_map<<8, z_deb=za; i<=yc; i++)
      {
	register long j,xmap_curr,ymap_curr;
	register double z_curr;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16); zbuf=ZBufferLine+(xdeb>>16);
	xmap_curr=xmap_deb; ymap_curr=ymap_deb; z_curr=z_deb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	    if(*zbuf++ > z_curr)
	    {
	      *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	      *(zbuf-1)=z_curr;
	    }
	  xmap_curr+=incXx; ymap_curr+=incYx; z_curr+=incZx;
	}
	xdeb+=pente3; xfin+=pente2;
	xmap_deb+=incX3; ymap_deb+=incY3; z_deb+=incZ3;
	ChunkyScreen+=HW_SCREEN_X_SIZE; ZBufferLine+=HW_SCREEN_X_SIZE;
      }
    }
    else
    {
      // trace la 2nde partie du triangle
      for(i=yb, xdeb=xb<<16, xfin=xa<<16, xmap_deb=xb_map<<8, ymap_deb=yb_map<<8, z_deb=zb; i<=yc; i++)
      {
	register long j,xmap_curr,ymap_curr;
	register double z_curr;
	if(i>=HW_SCREEN_Y_SIZE) return;
	cnk=ChunkyScreen+(xdeb>>16); zbuf=ZBufferLine+(xdeb>>16);
	xmap_curr=xmap_deb; ymap_curr=ymap_deb; z_curr=z_deb;
	for(j=xdeb>>16; j<=MIN(xfin>>16,HW_SCREEN_X_SIZE-1); j++)
	{
	  if((ymap_curr>>8)>=0 && (ymap_curr>>8)<256 && (xmap_curr>>8)>=0 && (xmap_curr>>8)<256)
	    if(*zbuf++ > z_curr)
	    {
	      *cnk++=Texture[((ymap_curr>>8)<<8)+(xmap_curr>>8)];
	      *(zbuf-1)=z_curr;
	    }
	  xmap_curr+=incXx; ymap_curr+=incYx; z_curr+=incZx;
	}
	xdeb+=pente2; xfin+=pente3;
	xmap_deb+=incX2; ymap_deb+=incY2; z_deb+=incZ2;
	ChunkyScreen+=HW_SCREEN_X_SIZE; ZBufferLine+=HW_SCREEN_X_SIZE;
      }
    }
  }
}
