
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "object.h"
#include "matrix.h"

/* Useful macros */
#define   intpar(str, x)   sscanf(str, "%*s %d", &(x))
#define   int3par(str, x1, x2, x3)   sscanf(str, "%*s %d%d%d", &(x1), &(x2), &(x3))
#define   int4par(str, x1, x2, x3, x4)   sscanf(str, "%*s %d%d%d%d", &(x1), &(x2), &(x3), &(x4))
#define   lngpar(str, x)   sscanf(str, "%*s %ld", &(x))
#define   dblpar(str, x)   sscanf(str, "%*s %lf", &(x))
#define   dbl3par(str, x, y, z)  sscanf(str, "%*s %lf%lf%lf", &(x), &(y), &(z))
#define   dbl4par(str, x1, x2, x3, x4)  sscanf(str, "%*s %lf%lf%lf%lf", &(x1), &(x2), &(x3), &(x4))
#define   dblpar(str, x)   sscanf(str, "%*s %lf", &(x))
#define   strpar(str, x)   sscanf(str, "%*s %s", (x))

#define MAX(a,b)         (((a) < (b)) ? b : a)

/* Forward declarations */
void VertexIndexesMoins1(Object *obj);
static void ComputeCameraToScreenTransformation(Camera *cam);
static void CenterObjectPoints(Object *obj);
static void CreateBBox(Object *obj, double Min_x, double  Min_y, double Min_z, 
                             double Max_x, double Max_y, double Max_z);


/*--------------------------------------*/
void ReadLight(char *filename, Light *Lumiere)
/*--------------------------------------*/
{
  FILE *fp;
  char  line[1024], command[1024];
  double x, y, z;
  int r, g, b;

  if ((fp = fopen (filename, "r")) == NULL) {
    fprintf (stderr, "Attention: file %s  can not open !!!\n", filename) ;
    exit(0);
  }

  SetMatrixToIdentity(&(Lumiere->LocalToGlobal));
  SetMatrixToIdentity(&(Lumiere->GlobalToLocal));

  while(fgets(line, 1024, fp) != NULL) 
    if((line[0] != '#') && (line[0] != 10)){
      if(sscanf(line, "%s", command) != 0) {

        if(!strcmp(command, "Position")) 
          /* Pos 3D du repere associe a la lumiere */
          dbl3par(line, x, y, z);

        if(!strcmp(command, "Type")) 
          intpar(line, Lumiere->Type);

        if(!strcmp(command, "Intensite")) 
          dblpar(line, Lumiere->intensite);

        if(!strcmp(command, "Angle")) 
          dblpar(line, Lumiere->angle);

        if(!strcmp(command, "Attenuation")) 
          dblpar(line, Lumiere->attenuation);

        if(!strcmp(command, "Color")) {
          /* Couleur de la lampe */
          int3par(line, r, g, b);
          Lumiere->color.r = (unsigned char) r;
          Lumiere->color.g = (unsigned char) g; 
          Lumiere->color.b = (unsigned char) b; 
        }

        if(!strcmp(command, "Puissance")) 
          intpar(line, Lumiere->puissance);

      }
    }
  fclose(fp);

  Lumiere->angle = cos(Lumiere->angle*M_PI/180);

  // Positionne la camera dans l'espace
  TranslateLight(Lumiere, x, y, z);
}


/*--------------------------------------*/
void ReadObject(char *filename, Object *obj)
/*--------------------------------------*/
/* Lit un fichier de description d'objet, met a jour les champs de la 
   structure objet */
{
  FILE *fp;
  char  line[1024], command[1024];
  int current_vertex, current_tri;
  double x, y, z, pitch, roll, yaw;
  short DeplacerIndicesDe_0_a_1 = 0;

  if ((fp = fopen (filename, "r")) == NULL) {
    fprintf (stderr, "Attention: file %s  can not open !!!\n", filename) ;
    exit(0);
  }

  SetMatrixToIdentity(&(obj->LocalToGlobal));
  SetMatrixToIdentity(&(obj->GlobalToLocal));
  SetMatrixToIdentity(&(obj->LocalToCamera));

  current_vertex = current_tri = 0;

  obj->material.color.r=240;
  obj->material.color.g=240;
  obj->material.color.b=20;
  while(fgets(line, 1024, fp) != NULL) 
    if(line[0] != '#') {
      if(sscanf(line, "%s", command) != 0) {

        if(!strcmp(command, "Position")) 
          /* Pos 3D du repere associe a l'objet */
          dbl3par(line, x, y, z);

        if(!strcmp(command, "Orientation")) 
          /* Orientation 3D du repere associe a l'objet */
          dbl3par(line, pitch, yaw, roll);

        /* Les sommets */
        if(!strcmp(command, "NbVertices")) {
          intpar(line, obj->NbVertices);
          /* alloc the room for all the vertices */
          obj->Vertex = (VERTEX *) malloc(obj->NbVertices*sizeof(VERTEX));
        }

        if(!strcmp(command, "Vertex")) {
          /* Read the vertex */
          dbl3par(line, 
                  obj->Vertex[current_vertex].ObjectVertex.x, 
                  obj->Vertex[current_vertex].ObjectVertex.y, 
                  obj->Vertex[current_vertex].ObjectVertex.z);
          current_vertex++;
        }

        /* Les faces */
        if(!strcmp(command, "NbTri")) {
          intpar(line, obj->NbTri);
          /* alloc the room for all the Triangular faces */
          obj->TriList = (Tri *) malloc(obj->NbTri * sizeof(Tri));
        }
  
        if(!strcmp(command, "Tri")) {
          /* Read the triangular face as vertex indexes */
          int3par(line, 
                  obj->TriList[current_tri].PtVertex[0],
                  obj->TriList[current_tri].PtVertex[1],
                  obj->TriList[current_tri].PtVertex[2]);
          current_tri++;
        }

        /* Indexs comemncent a 0 ou a 1 ? */
        if(!strcmp(command, "FirstIndexIs_1")) {
          DeplacerIndicesDe_0_a_1 = 1;
        }

        /* Caracteristiques de la surface */
        if(!strcmp(command, "Ka")) 
          dblpar(line, obj->material.Ka);

        if(!strcmp(command, "Kd")) 
          dblpar(line, obj->material.Kd);

        if(!strcmp(command, "Ks")) 
          dblpar(line, obj->material.Ks);

        if(!strcmp(command, "m")) 
          intpar(line, obj->material.m);
      }
    }
  fclose(fp);
  
  printf("NB Vertex Lus : %d, Nb Tri Lus : %d\n", current_vertex, current_tri);

  /* Petit hack pour lire des fichiers objets dont les indices des sommets 
     commencent a 1 et non a 0 */
  if(DeplacerIndicesDe_0_a_1)
    VertexIndexesMoins1(obj);

  // Transformation des points de l'objet pour que le repere objet se trouve au centre de l'objet
  CenterObjectPoints(obj);

  // Positionnement de l'objet a la position desiree
  RotateObject(obj,pitch,yaw,roll);
  TranslateObject(obj,x,y,z);

  // Calcul des normales aux faces
  precalculeNormales(obj);
}

/*-----------------------------------------*/
void VertexIndexesMoins1(Object *obj)
/*-----------------------------------------*/
/* Utile pour travailler avec des objets dont les indices de sommets 
   commencent a 1 et non a 0 */
{
  int i;

  for (i=0; i<obj->NbTri; i++)
  {
    obj->TriList[i].PtVertex[0]--;
    obj->TriList[i].PtVertex[1]--;
    obj->TriList[i].PtVertex[2]--;
  }
}



/*------------------------------------------*/
static void CenterObjectPoints(Object *obj)
/*------------------------------------------*/
/* Positionne le centre du repere objet sur le barycentre de l'objet  et 
   calcule la bounding box de l'objet */
 
{
  double Somme_x=0, Somme_y=0, Somme_z=0;
  double Min_x, Min_y, Min_z;
  double Max_x, Max_y, Max_z;
  int i;
  for (i=0; i < obj->NbVertices; i++){
    Somme_x += obj->Vertex[i].ObjectVertex.x;
    Somme_y += obj->Vertex[i].ObjectVertex.y;
    Somme_z += obj->Vertex[i].ObjectVertex.z;
  }
  Somme_x = (Somme_x / obj->NbVertices);
  Somme_y = (Somme_y / obj->NbVertices);
  Somme_z = (Somme_z / obj->NbVertices);
  
  Min_x = obj->Vertex[0].ObjectVertex.x - Somme_x;
  Min_y = obj->Vertex[0].ObjectVertex.y - Somme_y;
  Min_z = obj->Vertex[0].ObjectVertex.z - Somme_z;
  Max_x = Min_x; Max_y = Min_y; Max_z = Min_z;
  
  for (i=0; i < obj->NbVertices; i++){
     if ((obj->Vertex[i].ObjectVertex.x -= Somme_x) < Min_x)
      Min_x = obj->Vertex[i].ObjectVertex.x ;
    else
      if (obj->Vertex[i].ObjectVertex.x > Max_x)
        Max_x = obj->Vertex[i].ObjectVertex.x;

    if ((obj->Vertex[i].ObjectVertex.y -= Somme_y) < Min_y)
      Min_y = obj->Vertex[i].ObjectVertex.y ;
    else
      if (obj->Vertex[i].ObjectVertex.y > Max_y)
        Max_y = obj->Vertex[i].ObjectVertex.y;
 
   if ((obj->Vertex[i].ObjectVertex.z -= Somme_z) < Min_z)
      Min_z = obj->Vertex[i].ObjectVertex.z ;
    else
      if (obj->Vertex[i].ObjectVertex.z > Max_z)
        Max_z = obj->Vertex[i].ObjectVertex.z;
  }
  
  CreateBBox(obj, Min_x, Min_y, Min_z, Max_x, Max_y, Max_z);
}

/******************************************/
void CreateBBox(Object *obj, double Min_x, double  Min_y, double Min_z, 
                             double Max_x, double Max_y, double Max_z)
/******************************************/    
{
  obj->boundingBox.Vertex[0].ObjectVertex.x = Min_x;
  obj->boundingBox.Vertex[0].ObjectVertex.y = Min_y;
  obj->boundingBox.Vertex[0].ObjectVertex.z = Min_z;
    
  obj->boundingBox.Vertex[1].ObjectVertex.x = Min_x;
  obj->boundingBox.Vertex[1].ObjectVertex.y = Max_y;
  obj->boundingBox.Vertex[1].ObjectVertex.z = Min_z;
  
  obj->boundingBox.Vertex[2].ObjectVertex.x = Max_x;
  obj->boundingBox.Vertex[2].ObjectVertex.y = Max_y;
  obj->boundingBox.Vertex[2].ObjectVertex.z = Min_z;

  obj->boundingBox.Vertex[3].ObjectVertex.x = Max_x;
  obj->boundingBox.Vertex[3].ObjectVertex.y = Min_y;
  obj->boundingBox.Vertex[3].ObjectVertex.z = Min_z;

  obj->boundingBox.Vertex[4].ObjectVertex.x = Max_x;
  obj->boundingBox.Vertex[4].ObjectVertex.y = Min_y;
  obj->boundingBox.Vertex[4].ObjectVertex.z = Max_z;

  obj->boundingBox.Vertex[5].ObjectVertex.x = Min_x;
  obj->boundingBox.Vertex[5].ObjectVertex.y = Min_y;
  obj->boundingBox.Vertex[5].ObjectVertex.z = Max_z;

  obj->boundingBox.Vertex[6].ObjectVertex.x = Min_x;
  obj->boundingBox.Vertex[6].ObjectVertex.y = Max_y;
  obj->boundingBox.Vertex[6].ObjectVertex.z = Max_z;

  obj->boundingBox.Vertex[7].ObjectVertex.x = Max_x;
  obj->boundingBox.Vertex[7].ObjectVertex.y = Max_y;
  obj->boundingBox.Vertex[7].ObjectVertex.z = Max_z;

  obj->boundingBox.hauteur    = Max_x - Min_x;
  obj->boundingBox.largeur    = Max_y - Min_y;
  obj->boundingBox.profondeur = Max_z - Min_z;

  obj->boundingBox.plus_grand_cote_sur_2 =
    MAX(MAX(obj->boundingBox.hauteur, obj->boundingBox.largeur),obj->boundingBox.profondeur)/2.0;
}


/*                 1 ----------- 2
                    / |        /|
                   /  |      /  |
                6 ---------- 7  |
                  |   |     |   |
                  |   |     |   | 
                  |  0 --------/ 3
                  |   /     | /
                  | /       |/
                5 ----------- 4

*/


/*--------------------------------------*/
void ReadCamera(char *filename, Camera *obj)
/*--------------------------------------*/
{
  FILE *fp;
  char  line[1024], command[1024];
  double x, y, z, pitch, roll, yaw;

  if ((fp = fopen (filename, "r")) == NULL) {
    fprintf (stderr, "Attention: file %s  can not open !!!\n", filename) ;
    exit(0);
  }

  SetMatrixToIdentity(&(obj->LocalToGlobal));
  SetMatrixToIdentity(&(obj->GlobalToLocal));

  while(fgets(line, 1024, fp) != NULL) 
    if(line[0] != '#') {
      if(sscanf(line, "%s", command) != 0) {

        if(!strcmp(command, "Position")) 
          /* Pos 3D du repere associe a l'objet */
          dbl3par(line, x, y, z);

        if(!strcmp(command, "Orientation")) 
          /* Orientation 3D du repere associe a l'objet */
          dbl3par(line, pitch, yaw, roll);

        if(!strcmp(command, "fovX")) 
          /* Champ de vision en X */
          dblpar(line, obj->fovX);

        if(!strcmp(command, "fovY")) 
          /* Champ de vision en y */
          dblpar(line, obj->fovY);
        
        if(!strcmp(command, "resX")) 
          /* Resolution de l'ecran en X */
          intpar(line, obj->resX);

        if(!strcmp(command, "resY")) 
          /* Resolution de l'ecran en Y */
          intpar(line, obj->resY);
      }
    }
  fclose(fp);

  /* Positionne la camera dans l'espace */
  RotateCamera(obj, pitch, yaw, roll);
  TranslateCamera(obj, x, y, z);
  
  /* calcule les parametres camera : facteurs d'echelle et translation pour 
     passer vers le repere X11 */
  ComputeCameraToScreenTransformation(obj);
}

/*----------------------------------------------------------*/
static void ComputeCameraToScreenTransformation(Camera *cam)
/*----------------------------------------------------------*/
/* Calcul de cam->sx, cam->sy, cam->tx, cam->ty en fonction de cam->fovX, 
   cam->fovY, cam->resX et cam->resY */
{
  double tu, tv;
 
  /* on suppose la camera de focale 1 */
  tu = tan(cam->fovX*M_PI/180.);
  cam->sx = cam->resX/(2*tu);
 
  tv = tan(cam->fovY*M_PI/180.);
  cam->sy = cam->resY/(2*tv);
 
  cam->tx = cam->resX/2.;
  cam->ty = cam->resY/2.;
}

/*-------------------------------------------*/
void UpdateLocalToCamera(Object *obj, Camera *cam)
/*-------------------------------------------*/
/* Calcul de obj->LocalToCamera en fonction de cam->GlobalToLocal et 
   obj->LocalToGlobal */
{
  MultiplyMatrix(&cam->GlobalToLocal,&obj->LocalToGlobal,&obj->LocalToCamera);
}


/*-----------------------------------------------------------*/
void TranslateObject(Object *obj, double tx, double ty, double tz)
/*-----------------------------------------------------------*/
/* Applique une translation (tx, ty, tz) a l'objet. met a jour 
   obj->LocalToGlobal et obj->GlobalToLocal */
{
    Matrix4 T, T_Inv;

    GetTranslationMatrix(tx, ty, tz, &T, &T_Inv);
    MultiplyMatrix(&T, &(obj->LocalToGlobal), &(obj->LocalToGlobal));
    MultiplyMatrix(&(obj->GlobalToLocal), &T_Inv, &(obj->GlobalToLocal));
}

/*--------------------------------------------------------*/
void RotateObject(Object *obj, double rx, double ry, double rz)
/*--------------------------------------------------------*/
/* Applique a l'objet obj une rotation (rx, ry, rz) dans le repere du monde. 
   Met a jour obj->LocalToGlobal et obj->GlobalToLocal */
{
    Matrix4 R, R_Inv;

    GetRotationMatrixXYZ(rx, ry, rz, &R, &R_Inv);
    MultiplyMatrix( &R, &(obj->LocalToGlobal), &(obj->LocalToGlobal));
    MultiplyMatrix( &(obj->GlobalToLocal), &R_Inv, &(obj->GlobalToLocal));

}

/*------------------------------------------------------------------*/
void RotateObjectLocal(Object *obj, double rx, double ry, double rz)
/*------------------------------------------------------------------*/
/* Applique a l'objet obj une rotation (rx, ry, rz) dans son propre repere 
   (il tourne autours de ses propres axes). Met a jour obj->LocalToGlobal et 
   obj->GlobalToLocal */
{
  Matrix4 R, R_Inv;
 
  GetRotationMatrixXYZ(rx, ry, rz, &R, &R_Inv);
  
  MultiplyMatrix(&(obj->LocalToGlobal), &R, &(obj->LocalToGlobal));
  MultiplyMatrix(&R_Inv, &(obj->GlobalToLocal), &(obj->GlobalToLocal));
}

/*-----------------------------------------------------------*/
void TranslateCamera(Camera *obj, double tx, double ty, double tz)
/*-----------------------------------------------------------*/
/* Applique a la camera obj une translation (tx, ty, tz). met a jour 
   obj->LocalToGlobal et obj->GlobalToLocal */
{
    Matrix4 T, T_Inv;

    GetTranslationMatrix(tx, ty, tz, &T, &T_Inv);
    MultiplyMatrix( &T, &(obj->LocalToGlobal), &(obj->LocalToGlobal));
    MultiplyMatrix( &(obj->GlobalToLocal), &T_Inv, &(obj->GlobalToLocal));
}

/*--------------------------------------------------------*/
void RotateCamera(Camera *obj, double rx, double ry, double rz)
/*--------------------------------------------------------*/
/* Applique a la camera obj une rotation (rx, ry, rz) dans le repere du monde. 
   Met a jour obj->LocalToGlobal et obj->GlobalToLocal */
{
    Matrix4 R, R_Inv;

    GetRotationMatrixXYZ(rx, ry, rz, &R, &R_Inv);
    MultiplyMatrix(&R, &(obj->LocalToGlobal), &(obj->LocalToGlobal));
    MultiplyMatrix(&(obj->GlobalToLocal), &R_Inv, &(obj->GlobalToLocal));
}

/*------------------------------------------------------------------*/
void RotateCameraLocal(Camera *obj, double rx, double ry, double rz)
/*------------------------------------------------------------------*/
/* Applique a la camera obj une rotation (rx, ry, rz) dans son propre repere 
   (elle tourne autours de ses propres axes). Met a jour obj->LocalToGlobal et 
   obj->GlobalToLocal */
{
  Matrix4 R, R_Inv;
 
  GetRotationMatrixXYZ(rx, ry, rz, &R, &R_Inv);
  
  MultiplyMatrix(&(obj->LocalToGlobal), &R, &(obj->LocalToGlobal));
  MultiplyMatrix(&R_Inv, &(obj->GlobalToLocal), &(obj->GlobalToLocal));
}

/********************************************/
void PlaceCamera(Camera *cam, Object *obj)
{
/********************************************/
  double Ox, Oy, Oz, z;

  SetMatrixToIdentity(&cam->LocalToGlobal);
  SetMatrixToIdentity(&cam->GlobalToLocal);

  Ox = obj->LocalToGlobal.elem[0][3];
  Oy = obj->LocalToGlobal.elem[1][3];
  Oz = obj->LocalToGlobal.elem[2][3];

  z = Oz + obj->boundingBox.plus_grand_cote_sur_2 * (1 + 1/tan ((M_PI*cam->fovX) / 180.0)); 

  TranslateCamera(cam, Ox, Oy, z);
}


extern void precalculeNormales(Object *obj)
{
  int i;
  Tri *Faces;
  VERTEX *Points;
  COORD3D a,b,c,u,v;
  int T[obj->NbVertices];

  Faces  = obj->TriList;
  Points = obj->Vertex;

  // calcul des normales aux faces
  for(i=0; i<obj->NbTri; i++)
  {
    a=Points[Faces->PtVertex[0]].ObjectVertex;
    b=Points[Faces->PtVertex[1]].ObjectVertex;
    c=Points[Faces->PtVertex[2]].ObjectVertex;
    u=(COORD3D){b.x-a.x, b.y-a.y, b.z-a.z};
    v=(COORD3D){c.x-b.x, c.y-b.y, c.z-b.z};
    CrossProdPoint3D(&u,&v,&Faces->N);
    NormalizeVector3D(&Faces->N);
    Faces++;
  }

  // calcul des normales aux points
  Faces  = obj->TriList;
  for(i=0; i<obj->NbVertices; i++)
  {
    Points[i].N=(COORD3D){0,0,0};
    T[i]=0;
  }
  for(i=0; i<obj->NbTri; i++)
  {
    VERTEX *A,*B,*C;

    A=&Points[Faces->PtVertex[0]];
    B=&Points[Faces->PtVertex[1]];
    C=&Points[Faces->PtVertex[2]];

    A->N.x += Faces->N.x;
    A->N.y += Faces->N.y;
    A->N.z += Faces->N.z;

    B->N.x += Faces->N.x;
    B->N.y += Faces->N.y;
    B->N.z += Faces->N.z;

    C->N.x += Faces->N.x;
    C->N.y += Faces->N.y;
    C->N.z += Faces->N.z;

    T[Faces->PtVertex[0]]++;
    T[Faces->PtVertex[1]]++;
    T[Faces->PtVertex[2]]++;

    Faces++;
  }
  for(i=0; i<obj->NbVertices; i++)
  {
    Points[i].N.x /= T[i];
    Points[i].N.y /= T[i];
    Points[i].N.z /= T[i];
  }
}
