#include <sys/shm.h>
#include <X11/Xlib.h>
#include <X11/extensions/XShm.h>
#include <limits.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "object.h"
#include "graphics.h"
#include "param.h"
#include "virtualball.h"
#include "light.h"
#include "psb_draw.h"

#define SIGN(a) (((a) < 0) ? -1 : ((a==0) ? 0 : 1))

short mode_deplacement=AUCUN;

/* Forward declarations */
static void mainloop();
static void get_GC_black_and_white(Window win, GC *gc);
static void get_GC_color(Window win, GC *gc);
static void reset_GC_Colors();
static void clear_double_buffer();
static void display_double_buffer();
static void DrawFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D);
static void FillFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D, short color);
static void draw_object(Object *obj, Camera *cam, Light *light);
static void render_wire(Object *obj, Camera *cam);
static void render_fakelight(Object *obj, Camera *cam);
static void render_flatshading(Object *obj, Camera *cam, Light *light);
static void render_gouraud(Object *obj, Camera *cam, Light *light);
static void render_envmapping(Object *obj, Camera *cam, Light *light);
static void render_phong(Object *obj, Camera *cam, Light *light);
static void TransformSommets(Object *obj, Camera *cam);
static int  compare_z(const FaceToSort *f1, const FaceToSort *f2);
static void AllocateBuffers(Camera *cam);
static void close_graphics(Camera *cam);


/* Declarer une variable du type Display */
Display *display;

/* Specifier le nom du serveur X auquel on veut se connecter NULL
   signifie que l'on veut se connecter sur le serveur precise dans la
   variable d'environnement DISPLAY */
char *display_name=NULL;            

// Variable pour savoir si l'utilisateur deplace la lumiere ou l'objet
int movelight=0;

/* Declarer une variable contenant le numero d'ecran (screen) */
int screen;
Window win;                          /* descripteur de la fenetre */
Pixmap db,textzone;                  /* le double buffer */
GC gc, gcbg, gctext;                 /* ID du contexte graphique */
int width, height;                   /* taille de la fenetre */
int depth;
Colormap cmap;
long NColors;
int black, white;
Window root;
Visual *visual;

extern Object obj;
extern Camera cam;
extern Light light;
extern char *texture;
extern GlobalParams params;


// variables pour l'utilisation des ximages
XImage *Image;
Pixmap db;
Window win;
long Shm_Hard_Buffer_Flag;     /* flag et variable utiles pour l'allocation de Image 
				  en shared memory */
XShmSegmentInfo Shminfo;

/*----------------------------------------*/
void init_graphics(int Wsizex, int Wsizey)
/*----------------------------------------*/
/* Ouvre une fenetre graphique de taille Wsizex, Wsizey */
{
  unsigned int border_width=4;         /* largeur de la bordure */
  XSetWindowAttributes WAttribs;
  unsigned long WMask;

  if((display=XOpenDisplay(display_name))==NULL) {
    fprintf(stderr,"fenetre de base: ne peut pas se connecter ");
    fprintf(stderr,"au serveur %s\n",XDisplayName(display_name));
    exit(-1);
  }

  /* Initialiser la taille de la fenetre */
  width  = Wsizex;
  height = Wsizey;

  /* Recuperer les informations concernant l'ecran (root window) sur
     lequel on va afficher notre application. Le numero de l'ecran
     par defaut s'obtient a l'aide de la macro DefaultScreen() */
    
  screen = DefaultScreen(display);
  root   = RootWindow(display, screen);
  depth  = XDefaultDepth(display, screen);
  visual = DefaultVisual(display, screen);
  black  = BlackPixel(display, screen);
  white  = WhitePixel(display, screen);

  /* Allocation du double buffer */
  db = XCreatePixmap(display, root, width, height, depth);

  if(params.color_mode == BLACK_AND_WHITE_MODE) {
    /* creation d'une fenetre opaque a l'aide de la fonction
       XCreateSimpleWindow(). L'appel de XCreateSimpleWindow au lieu de
       XCreateWindow fait que la fenetre cree herite de tous les
       attributs de la fenetre parent */

    win = XCreateSimpleWindow(display, root, 0, 0,
                              width, height, border_width,
                              black, white);

    get_GC_black_and_white(win, &gc);
  }
  else {
    /* COLOR MODE */
    printf("Color mode on!\n");
    /* creation d'une colormap privee en niveaux de gris */
    cmap   = XCreateColormap(display, root, visual, AllocAll);
    Lut_GreyLevels(display, cmap);

    WAttribs.save_under = True;
    WAttribs.border_pixel = 0;
    WAttribs.colormap = cmap; 
    WAttribs.override_redirect = False;
    WAttribs.backing_store = Always;
                
    WMask= CWBorderPixel | CWBackingStore | CWSaveUnder | 
      CWOverrideRedirect | CWColormap ;

    win=XCreateWindow(display, 
                      root, 
                      0, 0, 
                      (unsigned int) width, (unsigned int) height,
                      0,
                      CopyFromParent,
                      CopyFromParent,
                      CopyFromParent,
                      WMask,
                      &WAttribs);

    get_GC_color(win, &gc);
  }

  /* selection des evenements que l'on desirera traiter pour cette fenetre */
  XSelectInput(display, win, ExposureMask | KeyPressMask |
               ButtonPressMask | ButtonReleaseMask | StructureNotifyMask | 
               PointerMotionMask);


  /* Pour le double buffer */
  gcbg=XCreateGC(display,root,0,NULL);
  XSetForeground(display,gcbg,white);

  /* affichage (mapping) de la fenetre, on mappe les differentes
     fenetres juste avant la gestion des evenements */

  XMapWindow(display,win);

  /* Allocation du pixmap pour la visu fil de fer et flat shading, et d'une 
     XImage pour la visu gouraud, phong et textures */
  AllocateBuffers(&cam);

  SetImageBuffer(&cam);

  mainloop();
} 


/*---------------------------------------*/
static void AllocateBuffers(Camera *cam)
/*---------------------------------------*/
/* Allocation des buffers de visualisation. Normalement, pour une scene 
   complete, il faudrait faire ca pour chaque camera.*/
{
  /* Pour la shared memory X11 */
  int CompletionType; 
  long Shm_OK;

  // Allocation du Z-Buffer
  cam->zbuffer = (double *)malloc(cam->resX * cam->resY * sizeof(double));

  /* Allocation du double buffer pour les modes de rendus simples */
  db = XCreatePixmap(display, root, cam->resX, cam->resY, depth);
  textzone=XCreatePixmap(display, root, 200, cam->resY, depth);

  /* Allocation du buffer cam->image, en Shared Memory X11 si possible */
  Shm_OK = XShmQueryExtension(display);

  switch(Shm_OK)
  {
  case 0:  
    /* On a pas pas pu allouer de Shared memory. On alloue donc le Z-Buffer 
       manuellement a l'aide d'un malloc */
    /* Allocation d'un buffer 8 bits */
    cam->image = (char *) malloc(cam->resX * cam->resY * sizeof(char));

    if (!cam->image)
    {
      perror("L'allocation de cam->image a echouee!\n");
      exit(0);
    }

    /* Allocation d'une XImage 8 bits */
    Image = XCreateImage(display, visual, 8, ZPixmap , 0, 
			 cam->image, cam->resX, cam->resY, 
			 8, 0);

    /* Flag qui indique qu'on a pas alloue la Shared Memory */
    Shm_Hard_Buffer_Flag = False;
    printf("Je n'ai pas reussi a allouer la Shared memory X11!\n");
    break;
  case 1: 
    /* On a pu allouer la Shared memory. Le Z-Buffer va donc se trouver dans 
       cette zone */
    Image = XShmCreateImage(display, visual ,8, ZPixmap , 
			    NULL, &Shminfo, cam->resX, cam->resY);

    Shminfo.shmid = shmget(IPC_PRIVATE, 
			   cam->resX * cam->resY * sizeof(char), 
			   IPC_CREAT | 0777);

    Shminfo.shmaddr = Image->data = cam->image = shmat(Shminfo.shmid, NULL, 0);

    if (!cam->image)
    {
      printf("Je n'ai pas reussi a allouer la Shared memory X11!\n");
      exit(0);
    }
    else
      printf("Shared memory X11 allouee!\n");

    Shminfo.readOnly = False;
    XShmAttach(display, &Shminfo);
    
    Shm_Hard_Buffer_Flag = True;
    CompletionType = XShmGetEventBase(display) + ShmCompletion; 
    break;  
  }

}

/*------------------------------------*/
static void close_graphics(Camera *cam)
/*------------------------------------*/
/* Libere les ressources allouees */
{
  free(cam->zbuffer);
  switch (Shm_Hard_Buffer_Flag)
  {
  case True:
    printf("Liberation de la memoire partagee\n");
    XShmDetach(display, &Shminfo);
    XDestroyImage(Image);
    shmdt(Shminfo.shmaddr);
    shmctl(Shminfo.shmid, IPC_RMID, 0);
    break;
  case False:
    /* Le buffer image */
    free(cam->image);
    XDestroyImage(Image);
    break;
  }
  XCloseDisplay(display); 
}


/*----------------------*/
static void mainloop()
/*----------------------*/
/* gestion des evenements:
   -prend un evenement
   -utilise le premier expose pour afficher texte et graphiques
   */
{
  XEvent report;                      /* structure des evenements */
  int old_x,old_y,new_x,new_y,dx,dy;
  double tx,ty,tz;

  while(1)
  {
    XNextEvent(display,&report);
    switch(report.type)
    {
      case Expose:
        /* on purge la queue des evenements de tous les Expose, pour
           ne redessiner la fenetre qu'une seule fois */
        while(XCheckTypedEvent(display,Expose,&report));
        draw_object(&obj, &cam, &light);
      break;

      case ConfigureNotify:
        break;

      case ButtonPress:
        old_x=report.xbutton.x;
        old_y=report.xbutton.y;
        switch(report.xbutton.button)
        {
          case Button1:
            mode_deplacement=ROTATION;
            break;
          case Button2:
            mode_deplacement=TRANSLATION;
            break;
          case Button3 :
            mode_deplacement=ZOOM;
            break;
        }
        break;

      case ButtonRelease:
        mode_deplacement=AUCUN;
        break;

      case KeyPress:
      {
        /* Les 4 lignes suivantes sont necessaires sous X11 pour detecter la
           valeur d'une touche entree au clavier dans une fenetre */
        char buffer;
        int bufsize=1;
        int charcount;
        charcount=XLookupString(&report,&buffer,bufsize,NULL,NULL);
        switch(buffer)
        {
          case 'q':
            // Quitte le programme
            close_graphics(&cam);
            exit(1);
            break;
	  case 'c':
	    // switch color<->grey
	    params.palette_mode^=1;
	    if(params.palette_mode==GREY) Lut_GreyLevels(display,cmap); else Lut_16M(display,cmap);
	    draw_object(&obj,&cam,&light);
	    break;
	  case 'b':
	    // Backspace culling
	    params.sort_mode=CULLING;
	    draw_object(&obj,&cam,&light);
	    printf("Sort mode = BACKSPACE CULLING\n");
	    break;
	  case 'z':
	    // Z-Buffer
	    params.sort_mode=ZBUFFER;
	    draw_object(&obj,&cam,&light);
	    printf("Sort mode = Z-BUFFER\n");
	    break;
          case 'w':
            // Visualisation fil de fer
            params.render_mode=WIREFRAME;
	    cam.render_mode=WIREFRAME;
	    cam.use_pixmap=1;
	    Lut_GreyLevels(display,cmap);
            reset_GC_Colors();
            draw_object(&obj,&cam,&light);
            printf("Render mode = WIREFRAME\n");
            break;
          case 'l':
            // Visualisation avec les faces cachees
            params.render_mode=HIDDEN_LINES;
	    cam.render_mode=HIDDEN_LINES;
	    cam.use_pixmap=1;
	    if(params.palette_mode==GREY) Lut_GreyLevels(display,cmap); else Lut_16M(display,cmap);
            reset_GC_Colors();
            draw_object(&obj,&cam,&light);
            printf("Render mode = HIDDEN_LINES\n");
            break;
          case 'f':
            // Visualisation en flat shading (eclairages)
            params.render_mode=FLAT_SHADING;
	    cam.render_mode=FLAT_SHADING;
	    cam.use_pixmap=1;
	    if(params.palette_mode==GREY) Lut_GreyLevels(display,cmap); else Lut_16M(display,cmap);
            reset_GC_Colors();
            draw_object(&obj,&cam,&light);
            printf("Render mode = FLAT_SHADING\n");
            break;
	  case 'p':
	    // Visualisation en phong
	    params.render_mode=PHONG;
	    cam.render_mode=PHONG;
	    cam.use_pixmap=0;
	    if(params.palette_mode==GREY) Lut_GreyLevels(display,cmap); else Lut_16M(display,cmap);
            reset_GC_Colors();
            draw_object(&obj,&cam,&light);
            printf("Render mode = PHONG\n");
            break;
          case 'o':
            // Visualisation en faux eclairage
            params.render_mode=FAKELIGHT;
	    cam.render_mode=FAKELIGHT;
	    cam.use_pixmap=0;
	    if(params.palette_mode==GREY) Lut_GreyLevels(display,cmap); else Lut_16M(display,cmap);
            reset_GC_Colors();
            draw_object(&obj,&cam,&light);
            printf("Render mode = FAKELIGHT\n");
            break;
	  case 'g':
	    // Visualisation en gouraud
	    params.render_mode=GOURAUD;
	    cam.render_mode=GOURAUD;
	    cam.use_pixmap=0;
	    if(params.palette_mode==GREY) Lut_GreyLevels(display,cmap); else Lut_16M(display,cmap);
	    reset_GC_Colors();
	    draw_object(&obj,&cam,&light);
	    printf("Render mode = GOURAUD\n");
	    break;
	  case 'e':
	    // Visualisation en environnement
	    params.render_mode=ENVMAPPING;
	    cam.render_mode=ENVMAPPING;
	    cam.use_pixmap=0;
	    SetEnvPalette(display,cmap);
	    reset_GC_Colors();
	    draw_object(&obj,&cam,&light);
	    printf("Render mode = ENVMAPPING\n");
	    break;
	  case '-':
	    movelight=1;
	    break;
	  case '+':
	    movelight=0;
	    break;
	  case '1':
	    if(obj.material.Ka<=0.9)
	      obj.material.Ka+=0.1;
	    else
	      obj.material.Ka=1;
	    obj.material.Ks=1-obj.material.Ka-obj.material.Kd;
	    if(obj.material.Ks<0)
	    {
	      obj.material.Ks=0;
	      obj.material.Kd=1-obj.material.Ka;
	    }
	    draw_object(&obj,&cam,&light);
	    break;
	  case '2':
	    if(obj.material.Kd<=0.9)
	      obj.material.Kd+=0.1;
	    else
	      obj.material.Kd=1;
	    obj.material.Ks=1-obj.material.Ka-obj.material.Kd;
	    if(obj.material.Ks<0)
	    {
	      obj.material.Ks=0;
	      obj.material.Ka=1-obj.material.Kd;
	    }
	    draw_object(&obj,&cam,&light);
	    break;
	  case '3':
	    if(obj.material.Ks<=0.9)
	      obj.material.Ks+=0.1;
	    else
	      obj.material.Ks=1;
	    obj.material.Ka=1-obj.material.Kd-obj.material.Ks;
	    if(obj.material.Ka<0)
	    {
	      obj.material.Ka=0;
	      obj.material.Kd=1-obj.material.Ks;
	    }
	    draw_object(&obj,&cam,&light);
	    break;
	  case '4':
	    if(obj.material.Ka>=0.1)
	      obj.material.Ka-=0.1;
	    else
	      obj.material.Ka=0;
	    obj.material.Ks=1-obj.material.Ka-obj.material.Kd;
	    if(obj.material.Ks<0)
	    {
	      obj.material.Ks=0;
	      obj.material.Kd=1-obj.material.Ka;
	    }
	    draw_object(&obj,&cam,&light);
	    break;
	  case '5':
	    if(obj.material.Kd>=0.1)
	      obj.material.Kd-=0.1;
	    else
	      obj.material.Kd=0;
	    obj.material.Ks=1-obj.material.Ka-obj.material.Kd;
	    if(obj.material.Ks<0)
	    {
	      obj.material.Ks=0;
	      obj.material.Ka=1-obj.material.Kd;
	    }
	    draw_object(&obj,&cam,&light);
	    break;
	  case '6':
	    if(obj.material.Ks>=0.1)
	      obj.material.Ks-=0.1;
	    else
	      obj.material.Ks=0;
	    obj.material.Ka=1-obj.material.Kd-obj.material.Ks;
	    if(obj.material.Ka<0)
	    {
	      obj.material.Ka=0;
	      obj.material.Kd=1-obj.material.Ks;
	    }
	    draw_object(&obj,&cam,&light);
	    break;
        }
        break;
      }

      case MotionNotify:
	/* on purge la queue des evenements de tous les MotionNotify, pour
	   ne redessiner la fenetre qu'une seule fois */
	while(XCheckTypedEvent(display, MotionNotify, &report));            
   
        new_x=report.xmotion.x;
        new_y=report.xmotion.y;
        /* dplacement en pixels */
        dx=new_x-old_x;
        dy=new_y-old_y;
        switch(mode_deplacement)
        {
          case TRANSLATION:
            tx=dx*3.0*obj.boundingBox.largeur/(double)cam.resX;
            ty=dy*3.0*obj.boundingBox.hauteur/(double)cam.resX;
	    if(movelight)
	    {
	      TranslateLight(&light,tx,ty,0);
	      LightUpdateLocalToCamera(&light,&cam);
	    }
	    else
	    {
	      TranslateCamera(&cam,tx,ty,0);
	      UpdateLocalToCamera(&obj,&cam);
	    }
            draw_object(&obj,&cam,&light);
            break;
          case ROTATION:
	    // On met l'objet a l'origine
	    tx=obj.LocalToGlobal.elem[0][3];
	    ty=obj.LocalToGlobal.elem[1][3];
	    tz=obj.LocalToGlobal.elem[2][3];
	    if(movelight)
	    {
	      TranslateLight(&light,-tx,-ty,-tz);
	      RotateLightVB(&cam,&light,old_x,old_y,new_x,new_y);
	      TranslateLight(&light,tx,ty,tz);
	      LightUpdateLocalToCamera(&light,&cam);
	    }
	    else
	    {
	      TranslateObject(&obj,-tx,-ty,-tz);
	      // Rotation de l'objet, boule virtuelle
	      RotateObjectVB(&cam,&obj,old_x,old_y,new_x,new_y);
	      // On remet l'objet la ou il etait
	      TranslateObject(&obj,tx,ty,tz);
	      UpdateLocalToCamera(&obj,&cam);
	    }
	    draw_object(&obj,&cam,&light);
            break;
          case ZOOM:
            tz=dy*3.0*obj.boundingBox.profondeur/(double)cam.resY;
            TranslateCamera(&cam,0,0,tz);
            UpdateLocalToCamera(&obj,&cam);
            draw_object(&obj,&cam,&light);
            break;
        }
        old_x=new_x;
        old_y=new_y;
        break;

      default:
        /* tous les evenements selectionnes par StructureNotifyMask
           (DestroyNotify, GravityNotify, etc...) sont filtres ici. */
        break;
    } 
  } 

}



/*----------------------------------------------------*/
static void get_GC_black_and_white(Window win, GC *gc)
/*----------------------------------------------------*/
/* Configure le contexte graphique pour du dessin en noir et blan 
   (couleur du trait = noir, fond = blanc */
{
  unsigned long valuemask = 0;                /* Ignore XGCvalues et prend les 
                                                 valeurs par defaut */
  XGCValues values;

  /* creation d'un contexte graphique par defaut */
  *gc = XCreateGC(display, win, valuemask, &values);

  /* specification d'un foreground noir */
  XSetForeground(display, *gc, black); 
}

/*-------------------------------------------*/
static void get_GC_color(Window win, GC *gc)
/*-------------------------------------------*/
/* COnfigure le contexte graphique pour un ecran couleur. Par defaut on 
   dessine en noir sur blanc */
{
  unsigned long valuemask = 0;                /* Ignore XGCvalues et prend les 
                                                 valeurs par defaut */
  XGCValues values;

  /* creation d'un contexte graphique par defaut */
  *gc = XCreateGC(display, win, valuemask, &values);

  /* specification d'un foreground blanc */
  XSetForeground(display, *gc, 255); 
}

/*--------------------*/
static void reset_GC_Colors()
/*--------------------*/
/* Remet la bonne valeur pour la couleur du dessin (noir en mode N/B, blanc 
   en mode couleur) */
{
  if(params.color_mode == BLACK_AND_WHITE_MODE)
    XSetForeground(display, gc, black);
  else
    XSetForeground(display, gc, 255); 
}

/*-----------------------------------------*/
static void clear_double_buffer(Camera *cam)
/*-----------------------------------------*/
/* Efface le contenu du double buffer */
{
  if(cam->use_pixmap)
    XFillRectangle(display, db, gcbg, 0, 0, cam->resX, cam->resY); 
  else 
    cam->image=(char *)memset(cam->image,0,sizeof(char)*cam->resX*cam->resY);
}

/*-------------------------------------------*/
static void display_double_buffer(Camera *cam)
/*-------------------------------------------*/
/* Recopie le double buffer dans la fenetre graphique */
{
  if(cam->use_pixmap)
    XCopyArea(display,db,win,gc,0,0,cam->resX,cam->resY,0,0); 
  else
    if(Shm_Hard_Buffer_Flag)
      XShmPutImage(display,win,gc,Image,0,0,0,0,cam->resX,cam->resY,NULL);  
    else
      XPutImage(display,win,gc,Image,0,0,0,0,cam->resX,cam->resY); 
  XFlush(display);
}

/*----------------------------------------------------------*/
static void DrawFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D)
/*----------------------------------------------------------*/
/* Dessine la face definie par les sommets A,B,C,D, en mode ligne.
   Si C == D alors il s'agit d'un triangle */
{ 
  XPoint point[5];

  point[0].x = A->x;
  point[0].y = A->y;
  point[1].x = B->x;
  point[1].y = B->y;
  point[2].x = C->x;
  point[2].y = C->y;

  if(C != D)
  {
    point[3].x = D->x;
    point[3].y = D->y;
    point[4].x = A->x;
    point[4].y = A->y;

    XDrawLines(display, db, gc, point, 5, CoordModeOrigin);
  }
  else
  {
    point[3].x = A->x;
    point[3].y = A->y;

    XDrawLines(display, db, gc, point, 4, CoordModeOrigin);
  }
}

/*-----------------------------------------------------------------------*/
static void FillFace(PIXEL *A, PIXEL *B, PIXEL *C, PIXEL *D, short color)
/*-----------------------------------------------------------------------*/
/* Dessine la face definie par les sommets A,B,C,D, en mode plein.
   Si C == D alors il s'agit d'un triangle */
{
  XPoint point[4];

  XSetForeground(display, gc, color);

  point[0].x = A->x;
  point[0].y = A->y;
  point[1].x = B->x;
  point[1].y = B->y;

  if(C != D)
  {
    point[2].x = C->x;
    point[2].y = C->y;
    point[3].x = D->x;
    point[3].y = D->y;

    XFillPolygon(display, db, gc, point, 4, Convex, CoordModeOrigin);
  }
  else
  {
    point[2].x = C->x;
    point[2].y = C->y;

    XFillPolygon(display, db, gc, point, 3, Convex, CoordModeOrigin);
  }
}


/*******************************************************/
void draw_boundingBox(Object *obj, Camera *cam)
{
/*******************************************************/
  int i;

  /* D'abord, projection des points */
  for (i=0; i<8; i++)
  {
    TransformPoint(&(obj->boundingBox.Vertex[i].ObjectVertex),
                   &(obj->LocalToCamera),
                   &(obj->boundingBox.Vertex[i].CameraVertex));

    obj->boundingBox.Vertex[i].ScreenVertex.x = ((obj->boundingBox.Vertex[i].CameraVertex.x / 
                                           obj->boundingBox.Vertex[i].CameraVertex.z) *
                                           cam->sx + cam->tx);
    obj->boundingBox.Vertex[i].ScreenVertex.y = ((obj->boundingBox.Vertex[i].CameraVertex.y / 
                                           obj->boundingBox.Vertex[i].CameraVertex.z) *
                                           cam->sy + cam->ty);                                   
  }

  /* Ensuite affichage des aretes de la bounding box */
  DrawFace(&(obj->boundingBox.Vertex[0].ScreenVertex),
           &(obj->boundingBox.Vertex[1].ScreenVertex),
           &(obj->boundingBox.Vertex[2].ScreenVertex),
           &(obj->boundingBox.Vertex[3].ScreenVertex));
  DrawFace(&(obj->boundingBox.Vertex[5].ScreenVertex),
           &(obj->boundingBox.Vertex[6].ScreenVertex),
           &(obj->boundingBox.Vertex[7].ScreenVertex),
           &(obj->boundingBox.Vertex[4].ScreenVertex));
  DrawFace(&(obj->boundingBox.Vertex[5].ScreenVertex),
           &(obj->boundingBox.Vertex[6].ScreenVertex),
           &(obj->boundingBox.Vertex[1].ScreenVertex),
           &(obj->boundingBox.Vertex[0].ScreenVertex));
  DrawFace(&(obj->boundingBox.Vertex[4].ScreenVertex),
           &(obj->boundingBox.Vertex[7].ScreenVertex),
           &(obj->boundingBox.Vertex[2].ScreenVertex),
           &(obj->boundingBox.Vertex[3].ScreenVertex));
}

/*-----------------------------------------------*/
static void draw_object(Object *obj, Camera *cam, Light *light)
/*-----------------------------------------------*/
/* Affiche dans la fenetre graphique l'objet passe en 1er parametre, comme 
   s'il etait vu par la camera passe en 2eme parametre */
{
  clear_double_buffer(cam); 

  // On met a jour les champs CameraVertex et ScreenVertex des sommets de l'objet pour pouvoir les dessiner
  TransformSommets(obj,cam);

  // dessin de la bounding box
  //  draw_boundingBox(obj,cam);

  switch(params.render_mode)
  {
    case WIREFRAME:
      render_wire(obj,cam);
      break;
    case FAKELIGHT:
      render_fakelight(obj,cam);
      break;
    case FLAT_SHADING:
      render_flatshading(obj,cam,light);
      break;
    case GOURAUD:
	render_gouraud(obj,cam,light);
      break;
    case ENVMAPPING:
      render_envmapping(obj,cam,light);
      break;
    case PHONG:
      render_phong(obj,cam,light);
      break;
  }
}

static void render_wire(Object *obj, Camera *cam)
{
  int i0,i1,i2,i3;
  int current_tri;
  double x,y,z,intensity;
  COORD3D NormCamera,visee;

  current_tri = obj->NbTri;
  while(current_tri--)
  {
    i0 = obj->TriList[current_tri].PtVertex[0];
    i1 = obj->TriList[current_tri].PtVertex[1];
    i2 = obj->TriList[current_tri].PtVertex[2];
    i3=i2;

    // recupere la normale a la face dans le repere camera
    TransformVector(&obj->TriList[current_tri].N,&obj->LocalToCamera,&NormCamera);
    NormalizeVector3D(&NormCamera);

    // calcule le vecteur de visee dans le repere camera
    x = -obj->Vertex[i0].CameraVertex.x;
    y = -obj->Vertex[i0].CameraVertex.y;
    z = -obj->Vertex[i0].CameraVertex.z;
    visee=(COORD3D){x,y,z};
    NormalizeVector3D(&visee);

    // calcule le produit scalaire normale.visee
    intensity=DotProdPoint3D(&NormCamera,&visee);

    if(intensity>0)
    {
      // la face est visible
      G_line(obj->Vertex[i0].ScreenVertex.x,obj->Vertex[i0].ScreenVertex.y,
	     obj->Vertex[i1].ScreenVertex.x,obj->Vertex[i1].ScreenVertex.y,255);
      G_line(obj->Vertex[i1].ScreenVertex.x,obj->Vertex[i1].ScreenVertex.y,
	     obj->Vertex[i2].ScreenVertex.x,obj->Vertex[i2].ScreenVertex.y,255);
      G_line(obj->Vertex[i2].ScreenVertex.x,obj->Vertex[i2].ScreenVertex.y,
	     obj->Vertex[i3].ScreenVertex.x,obj->Vertex[i3].ScreenVertex.y,255);
      G_line(obj->Vertex[i3].ScreenVertex.x,obj->Vertex[i3].ScreenVertex.y,
	     obj->Vertex[i0].ScreenVertex.x,obj->Vertex[i0].ScreenVertex.y,255);

      /*      DrawFace(&(obj->Vertex[i0].ScreenVertex),
	       &(obj->Vertex[i1].ScreenVertex),
	       &(obj->Vertex[i2].ScreenVertex),
	       &(obj->Vertex[i3].ScreenVertex));*/
    }
  }

  // recopie le double buffer dans la fenetre X-Window
  display_double_buffer(cam);
}


static void render_fakelight(Object *obj, Camera *cam)
{
  int i,i0,i1,i2,i3;
  int current_tri;
  FaceToSort sort_z[obj->NbTri];
  int nfacevis=0, nface=0;
  double x,y,z,intensity;
  COORD3D NormCamera,visee;

  current_tri = obj->NbTri;
  while(current_tri--)
  {
    i0 = obj->TriList[current_tri].PtVertex[0];
    i1 = obj->TriList[current_tri].PtVertex[1];
    i2 = obj->TriList[current_tri].PtVertex[2];
    i3=i2;

    // recupere la normale a la face dans le repere camera
    TransformVector(&obj->TriList[current_tri].N,&obj->LocalToCamera,&NormCamera);
    NormalizeVector3D(&NormCamera);

    // calcule le vecteur de visee dans le repere camera
    x = -obj->Vertex[i0].CameraVertex.x;
    y = -obj->Vertex[i0].CameraVertex.y;
    z = -obj->Vertex[i0].CameraVertex.z;
    visee=(COORD3D){x,y,z};
    NormalizeVector3D(&visee);

    // calcule le produit scalaire normale.visee
    intensity=DotProdPoint3D(&NormCamera,&visee);

    if(intensity>0)
    {
      // la face est visible
      sort_z[nfacevis].i=intensity;
      sort_z[nfacevis].z=obj->Vertex[i0].CameraVertex.z+
	                 obj->Vertex[i1].CameraVertex.z+
                         obj->Vertex[i2].CameraVertex.z;
      sort_z[nfacevis].face=current_tri;
      nfacevis++;
    }

    nface++;
  }

  /* On ne trie que si on veut faire l'elimination des faces cachees, et ca
     n'a pas de sens en fil de fer */
  qsort(sort_z,nfacevis,sizeof(FaceToSort),compare_z);

  if(cam->use_pixmap)
    for(i=0; i<nfacevis; i++)
    {
      Tri *t=&obj->TriList[sort_z[i].face]; // pointeur sur la face en cours

      i0=t->PtVertex[0];
      i1=t->PtVertex[1];
      i2=t->PtVertex[2];
      i3=i2;

      FillFace(&(obj->Vertex[i0].ScreenVertex),
	       &(obj->Vertex[i1].ScreenVertex),
	       &(obj->Vertex[i2].ScreenVertex),
	       &(obj->Vertex[i3].ScreenVertex),
	       255*sort_z[i].i);
    }
  else
    for(i=0; i<nfacevis; i++)
      G_ambient_polygon(&obj->TriList[sort_z[i].face],obj,255*sort_z[i].i);

  // recopie le double buffer dans la fenetre X-Window
  display_double_buffer(cam);
}


static void render_flatshading(Object *obj, Camera *cam, Light *light)
{
  int i,i0,i1,i2,i3;
  int current_tri;
  FaceToSort sort_z[obj->NbTri];
  int nfacevis=0, nface=0;
  double x,y,z,intensity;
  COORD3D NormCamera,visee;

  current_tri = obj->NbTri;
  while(current_tri--)
  {
    i0 = obj->TriList[current_tri].PtVertex[0];
    i1 = obj->TriList[current_tri].PtVertex[1];
    i2 = obj->TriList[current_tri].PtVertex[2];
    i3=i2;

    // recupere la normale a la face dans le repere camera
    TransformVector(&obj->TriList[current_tri].N,&obj->LocalToCamera,&NormCamera);
    NormalizeVector3D(&NormCamera);

    // calcule le vecteur de visee dans le repere camera
    x = -obj->Vertex[i0].CameraVertex.x;
    y = -obj->Vertex[i0].CameraVertex.y;
    z = -obj->Vertex[i0].CameraVertex.z;
    visee=(COORD3D){x,y,z};
    NormalizeVector3D(&visee);

    // calcule le produit scalaire normale.visee
    intensity=DotProdPoint3D(&NormCamera,&visee);

    if(intensity>0)
    {
      // la face est visible
      sort_z[nfacevis].i=intensity;
      sort_z[nfacevis].z=obj->Vertex[i0].CameraVertex.z+
	                 obj->Vertex[i1].CameraVertex.z+
                         obj->Vertex[i2].CameraVertex.z;
      sort_z[nfacevis].face=current_tri;
      nfacevis++;
    }

    nface++;
  }

  /* On ne trie que si on veut faire l'elimination des faces cachees, et ca
     n'a pas de sens en fil de fer */
  qsort(sort_z,nfacevis,sizeof(FaceToSort),compare_z);

  if(cam->use_pixmap)
    for(i=0; i<nfacevis; i++)
    {
      Tri *t;

      t=&obj->TriList[sort_z[i].face]; // pointeur sur la face en cours
      intensity=255*IntensityFace(obj,light,cam,t);
      i0=t->PtVertex[0];
      i1=t->PtVertex[1];
      i2=t->PtVertex[2];
      i3=i2;

      FillFace(&(obj->Vertex[i0].ScreenVertex),
	       &(obj->Vertex[i1].ScreenVertex),
	       &(obj->Vertex[i2].ScreenVertex),
	       &(obj->Vertex[i3].ScreenVertex),
	       intensity);
    }
  else
    for(i=0; i<nfacevis; i++)
    {
      Tri *t;

      t=&obj->TriList[sort_z[i].face]; // pointeur sur la face en cours
      intensity=255*IntensityFace(obj,light,cam,t);
      G_ambient_polygon(t,obj,intensity);
    }

  // recopie le double buffer dans la fenetre X-Window
  display_double_buffer(cam);
}


static void render_gouraud(Object *obj, Camera *cam, Light *light)
{
  int i,i0,i1,i2,i3;
  int current_tri;
  FaceToSort sort_z[obj->NbTri];
  int nfacevis=0, nface=0;
  double x,y,z,intensity;
  COORD3D NormCamera,visee;

  current_tri = obj->NbTri;
  while(current_tri--)
  {
    i0 = obj->TriList[current_tri].PtVertex[0];
    i1 = obj->TriList[current_tri].PtVertex[1];
    i2 = obj->TriList[current_tri].PtVertex[2];
    i3=i2;

    // recupere la normale a la face dans le repere camera
    TransformVector(&obj->TriList[current_tri].N,&obj->LocalToCamera,&NormCamera);
    NormalizeVector3D(&NormCamera);

    // calcule le vecteur de visee dans le repere camera
    x = -obj->Vertex[i0].CameraVertex.x;
    y = -obj->Vertex[i0].CameraVertex.y;
    z = -obj->Vertex[i0].CameraVertex.z;
    visee=(COORD3D){x,y,z};
    NormalizeVector3D(&visee);

    // calcule le produit scalaire normale.visee
    intensity=DotProdPoint3D(&NormCamera,&visee);

    if(intensity>0)
    {
      // la face est visible
      sort_z[nfacevis].i=intensity;
      sort_z[nfacevis].z=obj->Vertex[i0].CameraVertex.z+
	                 obj->Vertex[i1].CameraVertex.z+
                         obj->Vertex[i2].CameraVertex.z;
      sort_z[nfacevis].face=current_tri;
      nfacevis++;
    }

    nface++;
  }

  qsort(sort_z,nfacevis,sizeof(FaceToSort),compare_z);

  if(params.palette_mode==GREY)
    for(i=0; i<nfacevis; i++)
    {
      Tri *t;

      t=&obj->TriList[sort_z[i].face]; // pointeur sur la face en cours
      PSB_GouraudFill(t,obj,light,cam);
    }
  else
    for(i=0; i<nfacevis; i++)
    {
      Tri *t;

      t=&obj->TriList[sort_z[i].face]; // pointeur sur la face en cours
      PSB_GouraudColorFill(t,obj,light,cam);
    }

  // recopie le double buffer dans la fenetre X-Window
  display_double_buffer(cam);
}


static void render_phong(Object *obj, Camera *cam, Light *light)
{
  int i,i0,i1,i2,i3;
  int current_tri;
  FaceToSort sort_z[obj->NbTri];
  int nfacevis=0, nface=0;
  double x,y,z,intensity;
  double *zbuf=cam->zbuffer;
  COORD3D NormCamera,visee;

  current_tri = obj->NbTri;
  while(current_tri--)
  {
    i0 = obj->TriList[current_tri].PtVertex[0];
    i1 = obj->TriList[current_tri].PtVertex[1];
    i2 = obj->TriList[current_tri].PtVertex[2];
    i3=i2;

    // recupere la normale a la face dans le repere camera
    TransformVector(&obj->TriList[current_tri].N,&obj->LocalToCamera,&NormCamera);
    NormalizeVector3D(&NormCamera);

    // calcule le vecteur de visee dans le repere camera
    x = -obj->Vertex[i0].CameraVertex.x;
    y = -obj->Vertex[i0].CameraVertex.y;
    z = -obj->Vertex[i0].CameraVertex.z;
    visee=(COORD3D){x,y,z};
    NormalizeVector3D(&visee);

    // calcule le produit scalaire normale.visee
    intensity=DotProdPoint3D(&NormCamera,&visee);

    if(intensity>0)
    {
      // la face est visible
      sort_z[nfacevis].i=intensity;
      sort_z[nfacevis].z=obj->Vertex[i0].CameraVertex.z+
	                 obj->Vertex[i1].CameraVertex.z+
                         obj->Vertex[i2].CameraVertex.z;
      sort_z[nfacevis].face=current_tri;
      nfacevis++;
    }

    nface++;
  }

  qsort(sort_z,nfacevis,sizeof(FaceToSort),compare_z);

  for(i=0; i<nfacevis; i++)
  {
    Tri *t;

    t=&obj->TriList[sort_z[i].face]; // pointeur sur la face en cours
    PSB_PhongFill(t,obj,light,cam);
  }

  // recopie le double buffer dans la fenetre X-Window
  display_double_buffer(cam);
}


static void render_envmapping(Object *obj, Camera *cam, Light *light)
{
  int i,i0,i1,i2,i3;
  int current_tri;
  FaceToSort sort_z[obj->NbTri];
  int nfacevis=0, nface=0;
  double x,y,z,intensity;
  double *zbuf=cam->zbuffer;
  COORD3D NormCamera,visee;

  if(params.sort_mode==ZBUFFER)
  {
    // initialise le Z-Buffer
    for(i=0; i<cam->resX*cam->resY; i++) *zbuf++=0;

    for(i=0; i<obj->NbTri; i++)
      PSB_EnvZBufferFill(&obj->TriList[i],obj,cam,texture);
    display_double_buffer(cam);
    return;
  }

  current_tri = obj->NbTri;
  while(current_tri--)
  {
    i0 = obj->TriList[current_tri].PtVertex[0];
    i1 = obj->TriList[current_tri].PtVertex[1];
    i2 = obj->TriList[current_tri].PtVertex[2];
    i3=i2;

    // recupere la normale a la face dans le repere camera
    TransformVector(&obj->TriList[current_tri].N,&obj->LocalToCamera,&NormCamera);
    NormalizeVector3D(&NormCamera);

    // calcule le vecteur de visee dans le repere camera
    x = -obj->Vertex[i0].CameraVertex.x;
    y = -obj->Vertex[i0].CameraVertex.y;
    z = -obj->Vertex[i0].CameraVertex.z;
    visee=(COORD3D){x,y,z};
    NormalizeVector3D(&visee);

    // calcule le produit scalaire normale.visee
    intensity=DotProdPoint3D(&NormCamera,&visee);

    if(intensity>0)
    {
      // la face est visible
      sort_z[nfacevis].i=intensity;
      sort_z[nfacevis].z=obj->Vertex[i0].CameraVertex.z+
	                 obj->Vertex[i1].CameraVertex.z+
                         obj->Vertex[i2].CameraVertex.z;
      sort_z[nfacevis].face=current_tri;
      nfacevis++;
    }

    nface++;
  }

  qsort(sort_z,nfacevis,sizeof(FaceToSort),compare_z);

  for(i=0; i<nfacevis; i++)
  {
    Tri *t;

    t=&obj->TriList[sort_z[i].face]; // pointeur sur la face en cours
    PSB_EnvFill(t,obj,cam,texture);
  }

  // recopie le double buffer dans la fenetre X-Window
  display_double_buffer(cam);
}


static int compare_z(const FaceToSort *f1, const FaceToSort *f2)
{
  double z1,z2;

  z1=f1->z;
  z2=f2->z;

  return z1-z2;
}


/*------------------------------------------------------*/
static void TransformSommets(Object *obj, Camera *cam)
/*------------------------------------------------------*/
/* met a jour les champs CameraVertex et ScreenVertex de tous les sommets de 
   l'objet passe en 1er parametre */
{
     /* CODE A ECRIRE : 

        Pour chaque sommet, projeter, ranger dans le champ CameraVertex, puis
        mise a l'echelle et translation  pour passer des 
        unites utilisees dans le repere du monde aux unites X11 (pixels). 
        Translation car le repere X11 n'est pas centre au centre de la 
        fenetre de visu , il est centre en haut a gauche de la fenetre de 
        visualisation, X a droite, Y en bas 

        Fonctions a utiliser : TransformPoint() */
  int i;
  COORD3D *ptcam;

  for(i=0; i < obj->NbVertices; i++) {
    TransformPoint(&(obj->Vertex[i].ObjectVertex), &(obj->LocalToCamera), 
                   &(obj->Vertex[i].CameraVertex));
    /* Projection et mise a l'echelle et translation  pour passer des 
       unites utilisees dans le repere du monde aux unites X11 (pixels). 
       Translation car le repere X11 n'est pas centre au centre de la 
       fenetre de visu , il est centre en haut a gauche de la fenetre de 
       visualisation, X a droite, Y en bas */
    
    ptcam = &(obj->Vertex[i].CameraVertex);
    obj->Vertex[i].ScreenVertex.x = (short) 
      ((cam->sx * (ptcam->x / ptcam->z)) + cam->tx);
    obj->Vertex[i].ScreenVertex.y = (short) ((cam->sy * 
                                              (ptcam->y / ptcam->z)) + 
                                              cam->ty);
  }
}
