#include "main.h"
#include "texture.h"

TextureClass texture[128];
int CurTexture = 0;
int MaxTextures = 127;


int Filter;

/*************************************************************************
SetTextureFilter(int newfilter)

	This function is for setting the texture filter on all textures in 
	memory. It's good for when you add options to your game and want to
	give the user the ability to turn on/off filtering.

	This function also runs the ReSetupTextures() function, which loops
	through all textures and sets them up according to the current Filter.

*************************************************************************/
void SetTextureFilter(int newfilter) {
	if (newfilter >= 0 && newfilter <= 2) {
		Filter = newfilter;
		ReSetupTextures();
	}
}




/*************************************************************************
int LoadTexture(char *filename, bool ColorIsAvg, bool AlphaIsAvg)
	
	Loads a texture and returns the OpenGL index reference to that 
	texture. ColorIsAvg and AlphaIsAvg tell our LoadBitmapTexture()
	function how to load the texture.

*************************************************************************/
int LoadTexture(char *filename, bool ColorIsAvg, bool AlphaIsAvg) {
	int a;

	a = CurTexture;

	memset(&texture[a], 0, sizeof(texture[a]));
	LoadBitmapTexture(&texture[a], filename, ColorIsAvg, AlphaIsAvg, false);
	texture[a].Id = a;
	SetupTexture(&texture[a]);

	CurTexture++;
	return CurTexture - 1;
}



/*************************************************************************
void ReSetupTextures(), void ReSetupTexture()

	Loops through textures that have been loaded and sets them up again.
	This is for when you call change Bilinear, MipMap, or Repeat and you 
	want that to affect all textures in memory.

	You can do this for a single texture by calling ReSetupTexture().

*************************************************************************/
void ReSetupTextures() {
	int a=0;

	for (a=0; a<CurTexture; a++) {
		SetupTexture(&texture[a]);
	}
}


void ReSetupTexture(int t) {
	if (t > CurTexture || t < 0) return;
	SetupTexture(&texture[t]);
}





/*************************************************************************
void LoadBitmapTexture(TextureClass *texture, char *filename, 
					   bool ColorIsAvg, bool AlphaIsAvg, bool Reversed)

	Loads "filename" to the TextureClass "texture".
	
	ColorIsAvg - loads the texture in black and white by using the r g b
	values as an average for each pixel.
	
	AlphaIsAvg - sets the alpha values for the whole image to the average
	of the r g b values for each pixel. Used when we load font textures so
	that the fonts are antialiased.

	Reversed - loads the image from end to beginning, this is used for when
	we load fonts. Bitmaps are oddly stored in reverse, with the first pixels
	being at the end of the file.

	Note: we go in reverse for loading fonts because then the pixels are in
	the right order. Then we'll use a different function to loop through and
	chop the font texture into separate textures.

*************************************************************************/
void LoadBitmapTexture(TextureClass *texture, char *filename, bool ColorIsAvg, bool AlphaIsAvg, bool Reversed) {
	FILE * File;
	BITMAPFILEHEADER FileHeader;
	BITMAPINFOHEADER InfoHeader;
	DWORD Type;
	RGBTRIPLE rgb;
	int i, j, o=0;

	// open filename for reading
	File = fopen(filename, "rb");

	// read the header and type, then seek to the next position
	fread(&FileHeader, sizeof(FileHeader), 1, File);
	fread(&Type, sizeof(Type), 1, File);
	fseek(File, sizeof(FileHeader), SEEK_SET);

	// read the infoheader
	fread(&InfoHeader, sizeof(InfoHeader), 1, File);


	// set our texture width and height from what is read in the file
	texture->Width = InfoHeader.biWidth;
	texture->Height = InfoHeader.biHeight;

	// allocate enough memory to store the bitmap, then fill the memory with zeros
	// this size is width * height * 4
	// 4 means 1 byte for red, green, blue, and alpha
	texture->Image = (byte *) malloc(texture->Width * texture->Height * 4);
	memset(texture->Image, 0, texture->Width * texture->Height * 4);
	

	// if we are in reverse, set our current offset to the very end of the file
	if (Reversed) o = texture->Width * texture->Height * 4;

	// loop through loading pixel by pixel
	for (j=0; j < InfoHeader.biHeight; j++) {

		// if we're in reverse, go backwards one horizontal line of pixels
		if (Reversed) o -= 2 * (texture->Width * 4);

		for (i=0; i < InfoHeader.biWidth; i++) {

			// load the r g and b values from the file into an RGBTRIPLE structure
			fread(&rgb, sizeof(rgb), 1, File);

			// color is not averaged, so get the pixels normally
			if (!ColorIsAvg) {
				texture->Image[o+0] = rgb.rgbtRed;
				texture->Image[o+1] = rgb.rgbtGreen;
				texture->Image[o+2] = rgb.rgbtBlue;
			}
			// color is the average, so set the r g b values to the average of the three
			else {
				texture->Image[o+0] = (rgb.rgbtRed + rgb.rgbtGreen + rgb.rgbtBlue) / 3;
				texture->Image[o+1] = (rgb.rgbtRed + rgb.rgbtGreen + rgb.rgbtBlue) / 3;
				texture->Image[o+2] = (rgb.rgbtRed + rgb.rgbtGreen + rgb.rgbtBlue) / 3;
			}

			// Now we'll set the alpha value for this pixel. if AlphaIsAvg, we average
			// the r g and b to get alpha. Otherwise, check if the pixel is bright pink,
			// if so, set the pixel to completely transparent. If its normal, set alpha
			// to 255, meaning completely transparent.
			if (AlphaIsAvg)
				texture->Image[o+3] = ((rgb.rgbtRed + rgb.rgbtGreen + rgb.rgbtBlue) / 3);
			else if (!AlphaIsAvg && rgb.rgbtRed == 255 && rgb.rgbtGreen == 0 && rgb.rgbtBlue == 255)
				texture->Image[o+3] = 0;
			else
				texture->Image[o+3] = 255;

			o += 4;
		}
	}

	fclose(File);
}



/*************************************************************************
void SetupTexture(TextureClass *texture)
	
	Sets the OpenGL texture properties for the supplied texture object.
	This uses the Repeat, Bilinear, and MipMap bool values to tell OpenGL
	how to set up our texture properties.

*************************************************************************/
void SetupTexture(TextureClass *texture) {

	// tell OpenGL we want to use this texture's Id
	glBindTexture(GL_TEXTURE_2D, texture->Id);

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	if (Filter == TF_NONE) {
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	}
	else if (Filter == TF_BILINEAR) {
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);
	}
	else if (Filter == TF_TRILINEAR) {
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
		glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	}

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glTexImage2D(GL_TEXTURE_2D, 0, 4, texture->Width, texture->Height, 0, GL_RGBA, GL_UNSIGNED_BYTE, texture->Image );
	if (Filter != TF_NONE) gluBuild2DMipmaps( GL_TEXTURE_2D, 4, texture->Width, texture->Height, GL_RGBA, GL_UNSIGNED_BYTE, texture->Image );
}
