//
// Concrete types of operators
//

import Algebraic;
import Display;
import Topology;

class Operator_matrixNewRow extends UnitaryOperator
{
	// Constructor
	Operator_matrixNewRow(Context context)
	{
		super(context, 0);
	}

	// Insert a new line
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (!(node instanceof Operator_matrix)))
		{
			child = node;
			node = node.m_parent;
		}

		return (node==null
		        ? inseror
				: ((Operator_matrix)node).extendRow(child.m_rank));
	}
}

class Operator_matrixNewColumn extends UnitaryOperator
{
	// Constructor
	Operator_matrixNewColumn(Context context)
	{
		super(context, 0);
	}

	// Insert a new column
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (!(node instanceof Operator_matrix)))
		{
			child = node;
			node = node.m_parent;
		}

		return (node==null
		        ? inseror
				: ((Operator_matrix)node).extendColumn(child.m_rank));
	}
}

class Operator_matrixDeleteRow extends UnitaryOperator
{
	// Constructor
	Operator_matrixDeleteRow(Context context)
	{
		super(context, 0);
	}

	// Insert a new line
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (!(node instanceof Operator_matrix)))
		{
			child = node;
			node = node.m_parent;
		}

		return ((node!=null && ((Operator_matrix)node).getRows()>1)
				? ((Operator_matrix)node).deleteRow(child.m_rank)
				: inseror);
	}
}

class Operator_matrixDeleteColumn extends UnitaryOperator
{
	// Constructor
	Operator_matrixDeleteColumn(Context context)
	{
		super(context, 0);
	}

	// Insert a new line
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (!(node instanceof Operator_matrix)))
		{
			child = node;
			node = node.m_parent;
		}

		return ((node!=null && ((Operator_matrix)node).getColumns()>1)
				? ((Operator_matrix)node).deleteColumn(child.m_rank)
				: inseror);
	}
}

class Operator_matrixNextColumn extends UnitaryOperator
{
	// Constructor
	Operator_matrixNextColumn(Context context)
	{
		super(context, 0);
	}

	// Go to next column or create new column if does not exist
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (!(node instanceof Operator_matrix)))
		{
			child = node;
			node = node.m_parent;
		}

		if (node == null)
			return inseror;

		Operator_matrix matrix = (Operator_matrix)node;
		int column = matrix.columnOf(child.m_rank);

		if (column < matrix.getColumns()-1)
			return matrix.m_children[child.m_rank+1];
		else
			return matrix.extendColumn(child.m_rank);
	}
}

class Operator_matrixNextRow extends UnitaryOperator
{
	// Constructor
	Operator_matrixNextRow(Context context)
	{
		super(context, 0);
	}

	// Go to next column or create new column if does not exist
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (!(node instanceof Operator_matrix)))
		{
			child = node;
			node = node.m_parent;
		}

		if (node == null)
			return inseror;

		Operator_matrix matrix = (Operator_matrix)node;
		int row = matrix.rowOf(child.m_rank);

		if (row < matrix.getRows()-1)
			return matrix.m_children[matrix.indexOf(row+1, 0)];
		else
		{
			matrix.extendRow(child.m_rank);
			return matrix.m_children[matrix.indexOf(row+1, 0)];
		}
	}
}

class Operator_next extends UnitaryOperator
{
	// Constructor
	Operator_next(Context context)
	{
		super(context, 0);
	}

	// Return next argument in current unitary
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (!(node instanceof UnitaryOperator)))
		{
			child = node;
			node = node.m_parent;
		}

		return (node==null ? inseror : node.m_children[(child.m_rank+1)%node.m_children.length]);
	}
}

class Operator_jump extends UnitaryOperator
{
	// Constructor
	Operator_jump(Context context)
	{
		super(context, 0);
	}

	// Return next argument in current unitary
	public Operator flop(Operator inseror)
	{
		Operator node = inseror.m_parent;
		Operator child = inseror;

		while (node!=null && (node.m_breakers[child.m_rank]!=BREAK_CUT))
		{
			child = node;
			node = node.m_parent;
		}

		return (node==null ? inseror : node);
	}
}

class Operator_Pi extends PreConstantOperator
{
	// Constructor
	Operator_Pi(Context context)
	{
		super(context);
		m_display = new PiDisplay(this);
	}
}

class Operator_comma extends LeftBinaryOperator
{
	// Constructor
	Operator_comma(Context context)
	{
		super(context, 50, 50);
		m_display = new BinaryDisplay(this, ",");
	}
}

class Operator_equal extends RightBinaryOperator
{
	// Constructor
	Operator_equal(Context context)
	{
		super(context, 100, 100);
		m_display = new BinaryDisplay(this, "=");
	}
}

class Operator_plus extends LeftBinaryOperator
{
	// Constructor
	Operator_plus(Context context)
	{
		super(context, 200, 200);
		m_display = new BinaryDisplay(this, "+");
	}
}

class Operator_minus extends LeftBinaryOperator
{
	// Constructor
	Operator_minus(Context context)
	{
		super(context, 200, 200);
		m_display = new BinaryDisplay(this, "-");
	}
}

class Operator_multiply extends LeftBinaryOperator
{
	// Constructor
	Operator_multiply(Context context)
	{
		super(context, 300, 300);
		m_display = new BinaryDisplay(this, ".");
	}
}

class Operator_compose extends RightBinaryOperator
{
	// Constructor
	Operator_compose(Context context)
	{
		super(context, 380, 380);
		m_display = new ComposeDisplay(this);
	}
}

class Operator_superscript extends RightBinaryOperator
{
	// Constructor
	Operator_superscript(Context context)
	{
		super(context, 400, 450);
		m_display = new SuperscriptDisplay(this);
		m_breakers[0] = BREAK_SPLIT;
		m_breakers[1] = BREAK_CUT;
		Moving[] movings = new Moving[2];
		movings[0] = new Moving(0, Topology.UP, 1);
		movings[1] = new Moving(1, Topology.DOWN, 0);
		m_topology.setMovings(movings);
		m_topology.setEntering(0, 0, 0, 0);
	}
}

class Operator_subscript extends RightBinaryOperator
{
	// Constructor
	Operator_subscript(Context context)
	{
		super(context, 400, 450);
		m_display = new SubscriptDisplay(this);
		m_breakers[0] = BREAK_SPLIT;
		m_breakers[1] = BREAK_CUT;
		Moving[] movings = new Moving[2];
		movings[0] = new Moving(0, Topology.DOWN, 1);
		movings[1] = new Moving(1, Topology.UP, 0);
		m_topology.setMovings(movings);
		m_topology.setEntering(0, 0, 0, 0);
	}
}

class Operator_fraction extends LeftBinaryOperator
{
	// Constructor
	Operator_fraction(Context context)
	{
		super(context, 300, 300);
		m_display = new FractionDisplay(this);
		m_breakers[0] = BREAK_CUT;
		m_breakers[1] = BREAK_CUT;
		Moving[] movings = new Moving[2];
		movings[0] = new Moving(0, Topology.DOWN, 1);
		movings[1] = new Moving(1, Topology.UP, 0);
		m_topology.setMovings(movings);
		m_topology.setEntering(0, 0, 0, 1);
	}
}

class Operator_opposite extends LeftUnaryOperator
{
	// Constructor
	Operator_opposite(Context context)
	{
		super(context, 400, 450);
		m_display = new LeftUnaryDisplay(this, "-");
	}
}

class Operator_sqrt extends UnitaryOperator
{
	// Constructor
	Operator_sqrt(Context context)
	{
		super(context, 1);
		m_display = new SqrtDisplay(this);
		m_topology.setEntering(0, 0, 0, 0);
	}
}

class Operator_matrix extends UnitaryOperator
{
	// Constructor
	Operator_matrix(Context context)
	{
		super(context, 1);
		m_rows = 1;
		m_columns = 1;
		m_display = new MatrixDisplay(this);
		m_topology.setEntering(0, 0, 0, 0);
	}

	// Structural accessor
	public int getColumns()
	{
		return m_columns;
	}

	// Structural accessor
	public int getRows()
	{
		return m_rows;
	}

	// Structural accessor
	public int indexOf(int row, int column)
	{
		return column+row*m_columns;
	}

	// Structural accessor
	public int rowOf(int index)
	{
		return index / m_columns;
	}

	// Structural accessor
	public int columnOf(int index)
	{
		return index % m_columns;
	}

	// Display accessor
	public Display getChildDisplay(int row, int column)
	{
		return super.getChildDisplay(indexOf(row, column));
	}

	// Set movings by dynamic auto-calculation
	public void setMovings()
	{
		int index = 0;
		Moving[] movings = new Moving[2*(m_rows*(m_columns-1))+2*(m_columns*(m_rows-1))];

		for (int row=0; row<m_rows; ++row)
			for (int column=0; column<m_columns-1; ++column)
				movings[index++] = new Moving(indexOf(row, column), Topology.RIGHT, indexOf(row, column+1));

		for (int row=0; row<m_rows; ++row)
			for (int column=1; column<m_columns; ++column)
				movings[index++] = new Moving(indexOf(row, column), Topology.LEFT, indexOf(row, column-1));

		for (int row=0; row<m_rows-1; ++row)
			for (int column=0; column<m_columns; ++column)
				movings[index++] = new Moving(indexOf(row, column), Topology.DOWN, indexOf(row+1, column));

		for (int row=1; row<m_rows; ++row)
			for (int column=0; column<m_columns; ++column)
				movings[index++] = new Moving(indexOf(row, column), Topology.UP, indexOf(row-1, column));

		m_topology.setMovings(movings);
	}

	// Extend matrix from a row
	public Operator extendRow(int rank)
	{
		// Precondition: rank is supposed to be valid

		int rankRow = rowOf(rank);
		int rankColumn = columnOf(rank);

		// Reallocate for one more row of children
		Operator[] newChildren = new Operator[m_children.length+m_columns];
		for (int row=0; row<=rankRow; ++row)
		{
			for (int column=0; column<m_columns; ++column)
				newChildren[indexOf(row, column)] = m_children[indexOf(row, column)];
		}
		for (int row=rankRow+1; row<m_rows; ++row)
		{
			for (int column=0; column<m_columns; ++column)
			{
				newChildren[indexOf(row+1, column)] = m_children[indexOf(row, column)];
				newChildren[indexOf(row+1, column)].m_rank = indexOf(row+1, column);
			}
		}

		m_children = newChildren;
		++m_rows;
		for (int column=0; column<m_columns; ++column)
		{
			link(this, new TemplateOperator(m_context), indexOf(rankRow+1, column));
		}

		setMovings();

		// Reallocate for one more row of break cuts
		int[] newBreakers = new int[m_breakers.length+m_columns];
		for (int index=0; index<m_breakers.length+m_columns; ++index)
			newBreakers[index] = BREAK_CUT;
		m_breakers = newBreakers;

		return m_children[indexOf(rankRow+1, rankColumn)];
	}

	// Extend matrix from a column
	public Operator extendColumn(int rank)
	{
		// Precondition: rank is supposed to be valid

		int rankRow = rowOf(rank);
		int rankColumn = columnOf(rank);


		// Reallocate for one more column of children
		Operator[] newChildren = new Operator[m_children.length+m_rows];
		for (int row=0; row<m_rows; ++row)
		{
			for (int column=0; column<=rankColumn; ++column)
			{
				newChildren[column+row*(m_columns+1)] = m_children[indexOf(row, column)];
				newChildren[column+row*(m_columns+1)].m_rank = column+row*(m_columns+1);
			}
		}
		for (int row=0; row<m_rows; ++row)
		{
			for (int column=rankColumn+1; column<m_columns; ++column)
			{
				newChildren[column+1+row*(m_columns+1)] = m_children[indexOf(row, column)];
				newChildren[column+1+row*(m_columns+1)].m_rank = column+1+row*(m_columns+1);
			}
		}

		m_children = newChildren;
		++m_columns;
		for (int row=0; row<m_rows; ++row)
		{
			link(this, new TemplateOperator(m_context), indexOf(row, rankColumn+1));
		}

		setMovings();

		// Reallocate for one more column of break cuts
		int[] newBreakers = new int[m_breakers.length+m_rows];
		for (int index=0; index<m_breakers.length+m_rows; ++index)
			newBreakers[index] = BREAK_CUT;
		m_breakers = newBreakers;

		return m_children[indexOf(rankRow, rankColumn+1)];
	}

	// Delete a matrix row
	public Operator deleteRow(int rank)
	{
		// Precondition: rank is supposed to be valid

		int rankRow = rowOf(rank);
		int rankColumn = columnOf(rank);

		// Remove deleted children
		for (int column=0; column<m_columns; ++column)
			remove(m_children[indexOf(rankRow, column)]);

		// Reallocate for one less row of children
		Operator[] newChildren = new Operator[m_children.length-m_columns];
		for (int row=0; row<rankRow; ++row)
		{
			for (int column=0; column<m_columns; ++column)
				newChildren[indexOf(row, column)] = m_children[indexOf(row, column)];
		}
		for (int row=rankRow+1; row<m_rows; ++row)
		{
			for (int column=0; column<m_columns; ++column)
			{
				newChildren[indexOf(row-1, column)] = m_children[indexOf(row, column)];
				newChildren[indexOf(row-1, column)].m_rank = indexOf(row-1, column);
			}
		}

		m_children = newChildren;
		--m_rows;

		setMovings();

		// Reallocate for one more row of break cuts
		int[] newBreakers = new int[m_breakers.length-m_columns];
		for (int index=0; index<m_breakers.length-m_columns; ++index)
			newBreakers[index] = BREAK_CUT;
		m_breakers = newBreakers;

		return m_children[rankRow>=m_rows ? indexOf(rankRow-1, rankColumn) : indexOf(rankRow, rankColumn)];
	}

	// Delete a matrix column
	public Operator deleteColumn(int rank)
	{
		// Precondition: rank is supposed to be valid

		int rankRow = rowOf(rank);
		int rankColumn = columnOf(rank);

		// Remove deleted children
		for (int row=0; row<m_rows; ++row)
			remove(m_children[indexOf(row, rankColumn)]);

		// Reallocate for one less column of children
		Operator[] newChildren = new Operator[m_children.length-m_rows];
		for (int row=0; row<m_rows; ++row)
		{
			for (int column=0; column<rankColumn; ++column)
			{
				newChildren[column+row*(m_columns-1)] = m_children[indexOf(row, column)];
				newChildren[column+row*(m_columns-1)].m_rank = column+row*(m_columns-1);
			}
		}
		for (int row=0; row<m_rows; ++row)
		{
			for (int column=rankColumn+1; column<m_columns; ++column)
			{
				newChildren[(column-1)+row*(m_columns-1)] = m_children[indexOf(row, column)];
				newChildren[(column-1)+row*(m_columns-1)].m_rank = (column-1)+row*(m_columns-1);
			}
		}

		m_children = newChildren;
		--m_columns;

		setMovings();

		// Reallocate for one more row of break cuts
		int[] newBreakers = new int[m_breakers.length-m_rows];
		for (int index=0; index<m_breakers.length-m_rows; ++index)
			newBreakers[index] = BREAK_CUT;
		m_breakers = newBreakers;

		return m_children[rankColumn>=m_columns ? indexOf(rankRow, rankColumn-1) : indexOf(rankRow, rankColumn)];
	}

	// Structural information
	protected int m_rows;
	protected int m_columns;
}

class Operator_sigma extends UnitaryOperator
{
	// Constructor
	Operator_sigma(Context context)
	{
		super(context, 3);
		m_display = new SigmaDisplay(this);
		Moving[] movings = new Moving[6];
		movings[0] = new Moving(0, Topology.UP, 2);
		movings[1] = new Moving(0, Topology.DOWN, 1);
		movings[2] = new Moving(1, Topology.RIGHT, 0);
		movings[3] = new Moving(1, Topology.UP, 2);
		movings[4] = new Moving(2, Topology.RIGHT, 0);
		movings[5] = new Moving(2, Topology.DOWN, 1);
		m_topology.setMovings(movings);
		m_topology.setEntering(0, 0, 0, 0);
	}
}
