//
// Symbols drawing
//


import java.awt.*;

import Context;
import EditableString;
import GraphicBox;

abstract class Symbol extends GraphicBox
{
	// Invalidate symbol if necessary, after syntactic modification
	public void validate(int newX, int newY)
	{
		x = newX + relativeX;
		y = newY + relativeY;

		Rectangle newValidate = new Rectangle(x, y, width, height);

		if (m_validate == null)
		{
			m_validate = newValidate;
			m_valid = false;
			return;
		}

		m_valid = newValidate.equals(m_validate);

		if (! m_valid)
		{
			if (m_context.invalidate.isEmpty())
				m_context.invalidate.reshape(m_validate.x, m_validate.y, m_validate.width, m_validate.height);
			else
				m_context.invalidate.add(m_validate);
			}

		m_validate = newValidate;
	}

	public abstract void localDraw(Graphics graphics);

	public void draw(Graphics graphics)
	{
		if (m_valid && !m_validate.intersects(m_context.invalidate))
			return;
		localDraw(graphics);
	}

	// Area validate
	protected Rectangle m_validate;    // Last region of validation
	protected boolean m_valid;         // false if symbol needs to be redrawn

	// Drawing context
	protected Context m_context;
}

class TemplateSymbol extends Symbol
{
	// Constructor
	TemplateSymbol(Context context)
	{
		m_context = context;
	}

	// Drawing symbol
	public void localDraw(Graphics graphics)
	{
		graphics.setColor(m_context.foreground);
		graphics.fillRect(x+width/4, y+height/2, width*3/4, height/2);
	}

	// Calculate symbol dimensions
	public void calculate(LocalContext localContext)
	{
		Font font = new Font(localContext.fontName, Font.PLAIN, localContext.fontSize);
		FontMetrics fontMetrics = m_context.panel.getFontMetrics(font);

		width = fontMetrics.charWidth('#');
		height = fontMetrics.getAscent();
		yarn = height;
	}
}

class PiSymbol extends Symbol
{
	// Constructor
	PiSymbol(Context context)
	{
		m_context = context;
	}

	// Drawing symbol
	public void localDraw(Graphics graphics)
	{
		graphics.setColor(m_context.foreground);
		graphics.drawLine(x+width/4, y+height/2, x+width/4, y+height);
		graphics.drawLine(x+width*3/4, y+height/2, x+width*3/4, y+height);
		graphics.drawLine(x+width/8, y+height/2, x+width*7/8, y+height/2);
	}

	// Calculate symbol dimensions
	public void calculate(LocalContext localContext)
	{
		Font font = new Font(localContext.fontName, Font.PLAIN, localContext.fontSize);
		FontMetrics fontMetrics = m_context.panel.getFontMetrics(font);

		width = fontMetrics.charWidth('#');
		height = fontMetrics.getAscent();
		yarn = height;
	}
}

class StringSymbol extends Symbol
{
	// Constructor
	StringSymbol(Context context, EditableString editableString)
	{
		m_context = context;
		m_value = editableString;
	}

	// Draw string
	public void localDraw(Graphics graphics)
	{
		Font font = new Font(fontName, Font.PLAIN, fontSize);
		graphics.setFont(font);
		if (m_value.isIdentifier())
			graphics.setColor(m_context.identifierColor);
		else if (m_value.isNumber())
			graphics.setColor(m_context.numberColor);
		else if (m_value.isNamer())
			graphics.setColor(m_context.namerColor);
		else
			graphics.setColor(m_context.foreground);
		graphics.drawString(m_value.getValue(), x, y+height);
	}

	// Calculate symbol dimensions
	public void calculate(LocalContext localContext)
	{
		fontName = localContext.fontName;
		fontSize = localContext.fontSize;

		Font font = new Font(fontName, Font.PLAIN, fontSize);
		FontMetrics fontMetrics = m_context.panel.getFontMetrics(font);

		width = fontMetrics.stringWidth(m_value.getValue());
		height = fontMetrics.getAscent();
		yarn = height;
	}

	// Symbol value
	protected EditableString m_value;

	// Font information
	protected String fontName;
	protected int fontSize;
}

class FractionSymbol extends Symbol
{
	// Constructor
	FractionSymbol(Context context)
	{
		m_context = context;
	}

	// Drawing symbol
	public void localDraw(Graphics graphics)
	{
		graphics.setColor(m_context.foreground);
		graphics.drawLine(x, y, x+width, y);
	}

	// Calculate symbol dimensions
	public void calculate(LocalContext localContext, GraphicBox childUp, GraphicBox childDown)
	{
		width = Math.max(childUp.width, childDown.width);
		height = 0;
		yarn = height;
	}
}

class BracketsSymbol extends Symbol
{
	// Constructor
	BracketsSymbol(Context context)
	{
		m_context = context;
	}

	// Invalidate symbol if necessary, after syntactic modification
	public void validate(int newX, int newY)
	{
		x = newX + relativeX;
		y = newY + relativeY;

		Rectangle newValidate = new Rectangle(x, y, width, height);

		if (m_validate == null)
		{
			m_validate = newValidate;
			m_valid = false;
			return;
		}

		m_valid = newValidate.equals(m_validate);

		if (! m_valid)
		{
			// May have to partially invalidate symbol
			if (newValidate.x==m_validate.x && newValidate.y==m_validate.y && newValidate.height==m_validate.height)
			{
				Rectangle partial = new Rectangle(m_validate.x+m_validate.width-bracketWidth,
				                                  m_validate.y,
												  bracketWidth,
												  m_validate.height);

				if (m_context.invalidate.isEmpty())
					m_context.invalidate.reshape(partial.x, partial.y, partial.width, partial.height);
				else
					m_context.invalidate.add(partial);
			}

			// Or have to invalidate all area
			else
			{
				if (m_context.invalidate.isEmpty())
					m_context.invalidate.reshape(m_validate.x, m_validate.y, m_validate.width, m_validate.height);
				else
					m_context.invalidate.add(m_validate);
			}
		}

		m_validate = newValidate;
	}

	// Drawing symbol
	public void localDraw(Graphics graphics)
	{
		graphics.setColor(m_context.foreground);

		// Left bracket
		graphics.drawLine(x, y, x+bracketWidth, y);
		graphics.drawLine(x, y+height, x+bracketWidth, y+height);
		graphics.drawLine(x, y, x, y+height);

		// Right bracket
		graphics.drawLine(x+width, y, x+width-bracketWidth, y);
		graphics.drawLine(x+width, y+height, x+width-bracketWidth, y+height);
		graphics.drawLine(x+width, y, x+width, y+height);
	}

	// Calculate symbol dimensions
	public void calculate(LocalContext localContext, GraphicBox child)
	{
		width = child.width + 2 * GraphicBox.deltaWidth + 2 * bracketWidth;
		height = child.height;
		yarn = child.yarn;
	}

	// Static information
	public static final int bracketWidth = 3;
}

class SqrtSymbol extends Symbol
{
	// Constructor
	SqrtSymbol(Context context)
	{
		m_context = context;
	}

	// Invalidate symbol if necessary, after syntactic modification
	public void validate(int newX, int newY)
	{
		x = newX + relativeX;
		y = newY + relativeY;

		Rectangle newValidate = new Rectangle(x, y, width, height);

		if (m_validate == null)
		{
			m_validate = newValidate;
			m_valid = false;
			return;
		}

		m_valid = newValidate.equals(m_validate);

		if (! m_valid)
		{
			// May have to partially invalidate symbol
			if (newValidate.x==m_validate.x && newValidate.y==m_validate.y && newValidate.height==m_validate.height)
			{
				Rectangle partial = new Rectangle(m_validate.x+m_validate.width-1,
				                                  m_validate.y,
												  1,
												  m_validate.height);

				if (m_context.invalidate.isEmpty())
					m_context.invalidate.reshape(partial.x, partial.y, partial.width, partial.height);
				else
					m_context.invalidate.add(partial);
			}

			// Or have to invalidate all area
			else
			{
				if (m_context.invalidate.isEmpty())
					m_context.invalidate.reshape(m_validate.x, m_validate.y, m_validate.width, m_validate.height);
				else
					m_context.invalidate.add(m_validate);
			}
		}

		m_validate = newValidate;
	}

	// Drawing symbol
	public void localDraw(Graphics graphics)
	{
		graphics.setColor(m_context.foreground);

		graphics.drawLine(x, y+height/2, x+preSlash, y+height);
		graphics.drawLine(x+preSlash, y+height, x+preSlash, y);
		graphics.drawLine(x+preSlash, y, x+width, y);
		graphics.drawLine(x+width, y, x+width, y+GraphicBox.deltaHeight);
	}

	// Calculate symbol dimensions
	public void calculate(LocalContext localContext, GraphicBox child)
	{						
		width = child.width + 2 * GraphicBox.deltaWidth + preSlash;
		height = child.height + GraphicBox.deltaHeight;
		yarn = GraphicBox.deltaHeight + child.yarn;
	}

	// Static information
	public static final int preSlash = 5;
}

class SigmaSymbol extends Symbol
{
	// Constructor
	SigmaSymbol(Context context)
	{
		m_context = context;
	}
	// Drawing symbol
	public void localDraw(Graphics graphics)
	{
		graphics.setColor(m_context.foreground);

		graphics.drawLine(x, y, x+width, y);
		graphics.drawLine(x, y+height, x+width, y+height);
		graphics.drawLine(x, y, x+width/2, y+height/2);
		graphics.drawLine(x+width/2, y+height/2, x, y+height);
	}

	// Calculate symbol dimensions
	public void calculate(LocalContext localContext, GraphicBox child)
	{
		width = height = ((child.height-1) / stepSize + 1) * stepSize;
		yarn = height / 2;
	}

	// Static information
	public static final int stepSize = 4;
}