//
// Operator basic structure
//


import java.util.Stack;
import java.util.Vector;

import Algebraic;
import Context;
import Display;

class Cut
{
	// Constructor
	Cut(Operator newNode, Operator newChild, Stack newStack)
	{
		node = newNode;
		child = newChild;
		stack = newStack;
	}

	// Node and child of node
	public Operator node;
	public Operator child;
	public Stack stack;
}

abstract class Operator
{
	// Constructor
	Operator(Context context, int arity, int asInserorPriority, int asInserandPriority)
	{
		m_context = context;
		m_children = (arity!=0 ? new Operator[arity] : null);
		m_breakers = (arity!=0 ? new int[arity] : null);
		m_asInserorPriority = asInserorPriority;
		m_asInserandPriority = asInserandPriority;

		m_topology = new Topology();
	}

	// Parent <-> bidirectional primitive binding and garbage collecting
	public void basicLink(Operator parent, Operator child, int childRank)
	{
		// Child has new parent
		child.m_rank = childRank;
		child.m_parent = parent;

		// Parent has new child
		parent.m_children[childRank] = child;
	}

	// Parent <-> bidirectional binding and garbage collecting
	public void link(Operator parent, Operator child, int childRank)
	{
		if (child.m_asInserandPriority<parent.m_asInserorPriority &&
			parent.m_breakers[childRank]!=BREAK_CUT)
		{
			BracketsOperator brackets = new BracketsOperator(m_context);
			basicLink(parent, brackets, childRank);
			basicLink(brackets, child, 0);
		}
		else
			basicLink(parent, child, childRank);
	}

	// Remove an operator (self remove)
	public void remove(Operator node)
	{
		m_context.garbage.addElement(node);
	}

	// Find syntactical neighbor
	public Operator getSyntacticalNeighbor(int direction)
	{
		switch (direction)
		{
		case Topology.RIGHT :
			if (m_parent!=null && m_rank<m_parent.m_children.length-1)
				return m_parent.m_children[m_rank+1];
			break;

		case Topology.UP :
			return m_parent;
		
		case Topology.LEFT :
			if (m_parent!=null && m_rank>0)
				return m_parent.m_children[m_rank-1];
			break;
		
		case Topology.DOWN :
			if (m_children!=null)
				return m_children[0];
		}
		
		return null;
	}

	// Find node by topological entering
	protected Operator topologicalEnter(int direction)
	{
		int enteringChild = m_topology.getEntering(direction);

		return (enteringChild!=-1
		        ? m_children[enteringChild].topologicalEnter(direction)
				: this);
	}

	// Find topological neighbor
	public Operator getTopologicalNeighbor(int direction)
	{
		if (m_parent == null)
			return null;

		Moving moving = m_parent.m_topology.matchMoving(m_rank, direction);

		return (moving!=null
		        ? m_parent.m_children[moving.toChild].topologicalEnter(direction)
				: m_parent.getTopologicalNeighbor(direction));
	}

	// Insert "this" at "operator"
	public abstract Operator flop(Operator inseror);

	// Reconstruct by injection "this" for non linear insert
	public Operator reconstructByInjection(Stack stack, Operator destination, Operator inserand, int location, Operator continuation)
	{
		// Default function should never be called
		return destination;
	}

	// Reconstruct "this" for non linear insert
	public Operator reconstruct(Stack stack, Operator destination, Operator inserand, int location, Operator continuation)
	{
		Operator linear = destination;

		// Is there any split ?
		if (stack.empty() ||
			(m_breakers[((Operator)(stack.peek())).m_rank]==BREAK_SPLIT &&
			 inserand.m_asInserandPriority<m_asInserorPriority))
		{
			// Yes, then split
			if (location == Topology.RIGHT)
			{
				linear = flopAsIs(linear);
				linear = inserand.flop(linear);
			}
			else
			{
				linear = inserand.flop(linear);
				linear = flopAsIs(linear);
			}
			continuation.m_children[0] = linear;
			return linear;
		}
		
		// No, is there any cut ? (and arity > 0)
		else if (m_breakers[((Operator)(stack.peek())).m_rank] == BREAK_CUT &&
			     m_children.length > 0)
		{
			// Yes, then cut

			Operator child = (Operator)(stack.pop());
			int childRank = child.m_rank;
			Operator result = null;
			CollectorOperator collector = new CollectorOperator(m_context);
			
			// Reconstruct child injecting itself into new template
			result = child.reconstruct(stack, collector.m_children[0], inserand, location, continuation);
			
			// Replace child by the new template
			link(this, collector.m_children[0], childRank);

			// Inject this (with its reconstructed child) as is into linear
			linear = flopAsIs(linear);

			// Return
			return result;
		}

		// Else reconstruct by injection
		else
		{
			return reconstructByInjection(stack, destination, inserand, location, continuation);
		}
	}

	// Insert "as is" "operator"
	public Operator flopAsIs(Operator inseror)
	{
		link(inseror.m_parent, this, inseror.m_rank);
		return this;
	}

	// Delete node operation
	public Operator delete(int location)
	{
		if (location != Topology.RIGHT)
			return this;

		TemplateOperator templateOperator = new TemplateOperator(m_context);

		remove(this);
		link(m_parent, templateOperator, m_rank);
		return templateOperator;
	}

	// Delete child (default)
	public Operator deleteChild(int m_rank)
	{
		return m_children[m_rank];
	}

	// Find cut from a node ("this" is inserand, "operator" is first inseror)
	public Cut findCut(Operator operator)
	{
		Operator node = operator;
		Operator child = null;													   

		Stack stack = new Stack();
		stack.push(node);
		while (m_asInserandPriority < node.m_asInserorPriority)
		{
			child = node;
			node = node.m_parent;
			stack.push(node);
			if (node.m_breakers[child.m_rank] == BREAK_CUT)
				break;
		}

		return new Cut(node, child, stack);
	}

	// Display accessor
	public Display getChildDisplay(int index)
	{
		return (m_children!=null && index>=0 && index<m_children.length
		        ? m_children[index].m_display
				: null);
	}

	// Context accessor
	public Context getContext()
	{
		return m_context;
	}
 
	// Parent of node
	protected Operator m_parent;

	// Rank relative to this parent
	protected int m_rank;

	// Children of node
	protected Operator[] m_children;

	// Breakers
	protected int[] m_breakers;

	// Node topological information
	protected Topology m_topology;

	// Associated graphic display
	protected Display m_display;

	// Associated context
	protected Context m_context;

	// Priorities in the algebraic tree
	protected int m_asInserorPriority;
	protected int m_asInserandPriority;

	// Maximum value for priority
	public static final int maximumPriority = 32767;

	// Split/Cut values
	public static final int BREAK_NONE = 0;
	public static final int BREAK_SPLIT = 1;
	public static final int BREAK_CUT = 2;
}