//
// Formula
//


import java.awt.Color;
import java.awt.Graphics;
import java.awt.Panel;
import java.awt.Rectangle;
import java.util.Enumeration;
import java.util.Stack;
import java.util.Vector;

import Context;
import Display;
import GraphicBox;
import Operator;
import Topology;

abstract class Formula extends Operator
{
	// Constructor
	Formula(Context context, int x, int y)
	{
		super(context, 1, 0, Operator.maximumPriority);
		m_display = new FormulaDisplay(this, x, y);

		// Create first child of formula
		m_children[0] = new TemplateOperator(m_context);
		link(this, m_children[0], 0);

		// Create local context initiator
		m_localContext = new LocalContext(m_context.fontName, m_context.defaultFontSize);
	}

	// Specific command action on the formula
	public abstract void executeCommand(Command command);

	// New topological entering
	public Operator topologicalEnter(int directionInto)
	{
		return this;
	}

	// Redraw formula
	public void redraw(Graphics graphics, boolean partialRefresh)
	{
		Rectangle saveInvalidate = new Rectangle(m_context.invalidate.x,
		                                         m_context.invalidate.y,
		                                         m_context.invalidate.width,
												 m_context.invalidate.height);

		if (! partialRefresh)
		{
			// Not shared copy
			m_context.invalidate.x = m_context.bounds.x;
			m_context.invalidate.y = m_context.bounds.y;
			m_context.invalidate.width = m_context.bounds.width;
			m_context.invalidate.height = m_context.bounds.height;
		}

		// Delete invalid area
		if (! m_context.invalidate.isEmpty())
		{
			Color oldColor = graphics.getColor();
			graphics.setColor(m_context.background);
			graphics.fillRect(m_context.invalidate.x,
			                  m_context.invalidate.y,
							  m_context.invalidate.width+1,
							  m_context.invalidate.height+1);
			graphics.setColor(oldColor);
		}
 
		// Draw formula (only invalid symbols, or valid symbols intersecting invalid area)
		m_display.draw(graphics);

		if (! partialRefresh)
		{
			// Not shared copy
			m_context.invalidate.x = saveInvalidate.x;
			m_context.invalidate.y = saveInvalidate.y;
			m_context.invalidate.width = saveInvalidate.width;
			m_context.invalidate.height = saveInvalidate.height;
		}
	}

	// Update
	public void update()
	{
		m_context.invalidate.height = -1;  // Empty invalid area
		m_display.calculate(m_localContext);
		m_display.validate(m_display.x, m_display.y);
		purgeGarbage();
	}

	// Purge garbage
	protected void purgeGarbage()
	{
		Display element;

		for (Enumeration index=m_context.garbage.elements(); index.hasMoreElements();)
		{
			element = ((Operator)(index.nextElement())).m_display;

			// Extend invalid area
			if (m_context.invalidate.isEmpty())
				m_context.invalidate.reshape(element.x, element.y, element.width, element.height);
			else
				m_context.invalidate.add((Rectangle)element);
		}

		// Clear garbage
		m_context.garbage.removeAllElements();
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		// Should never be here (formula should not be an inserand)
		return inseror;
	}
														 
	// Reconstruct "this" for non linear insert
	public Operator reconstruct(Stack stack, Operator destination, Operator inserand, int location, Operator continuation)
	{
		return destination;  // No insertion
	}

	// Non linear insert (return new node of insert continuation)
	public Operator insert(Operator inserand, Operator inseror, int location)
	{
		if (inseror == null)
			return null;

		// New insert node for continuation
		Operator continuation = new CollectorOperator(m_context);

		// Find cut (and cut path)
		Cut cut = inserand.findCut(inseror);

		// Create new cut tree replacement
		Operator collector = new CollectorOperator(m_context);
		cut.stack.pop();
		if (cut.node.m_parent != null)
		{
			Operator parent = cut.node.m_parent;
			int rank = cut.node.m_rank;
			cut.node.reconstruct(cut.stack, collector.m_children[0], inserand, location, continuation);
			link(parent, collector.m_children[0], rank);
		}
		else
		{
			// In fact, cut.node is the formula, then reconstruct only its child
			int rank = cut.child.m_rank;
			cut.stack.pop();
			cut.child.reconstruct(cut.stack, collector.m_children[0], inserand, location, continuation);
			link(cut.node, collector.m_children[0], rank);
		}

		return continuation.m_children[0];
	}

	// Local context initiator
	protected LocalContext m_localContext;
}