//
// Editable formula
//


import java.awt.Graphics;

import Algebraic;
import Command;
import Completion;
import Context;
import Formula;
import Operator;
import Selection;
import Topology;
import Types;

class EditableFormula extends Formula
{
	EditableFormula(Context context, int x, int y)
	{
		super(context, x, y);

		m_selection = new Selection(m_children[0]);
		m_hasOnlySelectionChanged = false;
		update();
	}

	// Redraw formula
	public void redraw(Graphics graphics, boolean partialRefresh)
	{
		// Clear selection
		m_selection.clear(graphics);

		// Draw formula
		if (!m_hasOnlySelectionChanged || !partialRefresh)
			super.redraw(graphics, partialRefresh);

		// Draw selection
		m_selection.draw(graphics);
	}

	// Specific command action on the formula (the main automaton)
	public void executeCommand(Command command)
	{	
		m_hasOnlySelectionChanged = false;

		switch (m_state)
		{
		case m_stateExpression :
			doExpression(command);
			break;
		case m_stateCompletion :
			doCompletion(command);
			break;
		}

		if (! m_hasOnlySelectionChanged)
			update();
	}

	// Do expression input
	protected void doExpression(Command command)
	{
		switch (command.getAction())
		{

		// MOVINGS
		case Command.CONTROL_RIGHT :
			syntacticalMove(Topology.RIGHT);
			break;
		case Command.CONTROL_UP :
			syntacticalMove(Topology.UP);
			break;
		case Command.CONTROL_LEFT :
			syntacticalMove(Topology.LEFT);
			break;
		case Command.CONTROL_DOWN :
			syntacticalMove(Topology.DOWN);
			break;
		case Command.RIGHT :
			topologicalMove(Topology.RIGHT);
			break;
		case Command.UP :
			topologicalMove(Topology.UP);
			break;
		case Command.LEFT :
			topologicalMove(Topology.LEFT);
			break;
		case Command.DOWN :
			topologicalMove(Topology.DOWN);
			break;

		// DELETION
		case Command.DELETE :
			delete();
			break;

		// COMPLETION
		case Command.TAB :
			m_state = m_stateCompletion;
			doCompletion(new Command(Command.COMPLETE));
			break;
	
		// INSERTION (STANDARD OPERATOR)
		case Command.LETTER_OR_DIGIT :
			insertLetterOrDigit(command.getCharValue());
			break;
		case Command.COMMA :
			insertOperator(new Operator_comma(m_context));
			break;
		case Command.EQUAL :
			insertOperator(new Operator_equal(m_context));
			break;
		case Command.PLUS :
			insertOperator(new Operator_plus(m_context));
			break;
		case Command.MINUS :
			insertMinusOrOpposite();
			break;
		case Command.MULTIPLY :
			insertOperator(new Operator_multiply(m_context));
			break;
		case Command.FRACTION :
			insertOperator(new Operator_fraction(m_context));
			break;
		case Command.COMPOSE :
			insertOperator(new Operator_compose(m_context));
			break;
		case Command.SUBSCRIPT :
			insertOperator(new Operator_subscript(m_context));
			break;
		case Command.SUPERSCRIPT :
			insertOperator(new Operator_superscript(m_context));
			break;

		// INSERTION (BRACKETS)
		case Command.BRACKETS :
			insertBrackets();
			break;			

		// INSERTION (NAMER)
		case Command.SIGMA :
			insertOperator(new Operator_sigma(m_context));
			break;
		case Command.SQRT :
			insertOperator(new Operator_sqrt(m_context));
			break;
		case Command.MATRIX :
			insertOperator(new BracketsOperator(m_context));
			insertOperator(new Operator_matrix(m_context));
			break;
		case Command.PI :
			insertOperator(new Operator_Pi(m_context));
			break;

		// CONTROLS
		case Command.NEXT :  // Next argument
			m_selection.changeNode((new Operator_next(m_context)).flop(m_selection.node));
			m_hasOnlySelectionChanged = true;
			break;
		case Command.JUMP :  // Go outside unitary
			m_selection.changeNode((new Operator_jump(m_context)).flop(m_selection.node));
			m_hasOnlySelectionChanged = true;
			break;
		case Command.ENTER :
			insertEnter();
			break;
		case Command.OR :
			m_selection.changeNode((new Operator_matrixNewRow(m_context)).flop(m_selection.node));
			break;
		case Command.AND :
			m_selection.changeNode((new Operator_matrixNewColumn(m_context)).flop(m_selection.node));
			break;
		case Command.CONTROL_ROW :
			m_selection.changeNode((new Operator_matrixDeleteRow(m_context)).flop(m_selection.node));
			break;
		case Command.CONTROL_COLUMN :
			m_selection.changeNode((new Operator_matrixDeleteColumn(m_context)).flop(m_selection.node));
			break;
		}
	}

	// Insert letter or digit
	protected void insertLetterOrDigit(char value)
	{
		// Insert on a template
		if (m_selection.node instanceof TemplateOperator)
			m_selection.change(insert(new ConstantOperator(m_context, value), m_selection.node, Topology.RIGHT), Topology.RIGHT);

		// Or insert on a constant (then extend its value)
		else if (m_selection.node instanceof ConstantOperator &&
		         m_selection.location==Topology.RIGHT)
		{
			ConstantOperator constantOperator = (ConstantOperator)m_selection.node;
			m_selection.changeNode(constantOperator.extend(value));
		}
	}

	// Insert minus or opposite
	protected void insertMinusOrOpposite()
	{
		// Unary when at left position
		if (m_selection.location == Topology.LEFT)
			insertOperator(new Operator_opposite(m_context));
		// Binary when at right position
		else
			insertOperator(new Operator_minus(m_context));
	}

	// Insert brackets
	protected void insertBrackets()							  
	{
		// Brackets at right of any identifier generates a function
		if (m_selection.node instanceof ConstantOperator &&
			((ConstantOperator)(m_selection.node)).getValue().isIdentifier() &&
			m_selection.location==Topology.RIGHT)
			insertOperator(new Operator_compose(m_context));

		insertOperator(new BracketsOperator(m_context));
	}

	// Insert "enter" control
	protected void insertEnter()
	{
		Operator node;

		// May be on a namer
		if (m_selection.node instanceof ConstantOperator &&
			((ConstantOperator)(m_selection.node)).getValue().isNamer())
		{
			String namer = ((ConstantOperator)(m_selection.node)).getValue().getValue();

			// Name a composition
			if (m_selection.node.m_parent instanceof Operator_compose)
			{
				// Remove lost nodes
				remove(m_selection.node);
				remove(m_selection.node.m_parent);

				// Delete composition
				Operator parent = m_selection.node.m_parent.m_parent;
				Operator child = m_selection.node.m_parent.m_children[(m_selection.node.m_rank+1)%2];
				int rank = m_selection.node.m_parent.m_rank;
				link(parent, child, rank);

				// Insert named operator
				m_selection.changeNode(child);
				doExpression(new Command(namer));
			}

			// Isolated namer
			else
			{
				// Replace namer by a template
				m_selection.changeNode(insert(new TemplateOperator(m_context), m_selection.node, m_selection.location));

				// Insert named operator
				doExpression(new Command(namer));
			}
		}
	}

	// Syntactical moving
	protected void syntacticalMove(int direction)
	{
		Operator node = m_selection.node.getSyntacticalNeighbor(direction);
		if (node!=null && node!=this)
			m_selection.changeNode(node);
		m_hasOnlySelectionChanged = true;
	}

	// Topological moving
	protected void topologicalMove(int direction)
	{
		Operator node;

		switch (direction)
		{
		
		case Topology.RIGHT :
			if (m_selection.location == Topology.LEFT)
				m_selection.change(m_selection.node, Topology.RIGHT);
			else
			{
				node = m_selection.node.getTopologicalNeighbor(Topology.RIGHT);
				if (node != null)
					m_selection.change(node, Topology.LEFT);
			}
			break;

		case Topology.UP :
			node = m_selection.node.getTopologicalNeighbor(Topology.UP);
			if (node != null)
				m_selection.changeNode(node);
			break;

		case Topology.LEFT :
			if (m_selection.location == Topology.RIGHT)
				m_selection.change(m_selection.node, Topology.LEFT);
			else
			{
				node = m_selection.node.getTopologicalNeighbor(Topology.LEFT);
				if (node != null)
					m_selection.change(node, Topology.RIGHT);
			}
			break;

		case Topology.DOWN :
			node = m_selection.node.getTopologicalNeighbor(Topology.DOWN);
			if (node != null)
				m_selection.changeNode(node);
			break;
		}

		m_hasOnlySelectionChanged = true;
	}

	// Delete on selection
	protected void delete()
	{
		m_selection.changeNode(m_selection.node.delete(m_selection.location));
	}

	// Insert a new operator
	protected void insertOperator(Operator operator)
	{
		m_selection.changeNode(insert(operator, m_selection.node, m_selection.location));
	}

	// Do completion input
	protected void doCompletion(Command command)
	{
		switch (command.getAction())
		{
		// BEGIN COMPLETION
		case Command.COMPLETE :
			beginCompletion();
			break;

		// CONTINUE PROCESS (CIRCULARY)
		case Command.TAB :
			continueCompletion();
			break;

		// TERMINATE OR BREAK COMPLETION PROCESS
		default :
			m_completion = null;
			m_state = m_stateExpression;
			doExpression(command);
		}
	}

	// Begin completion
	protected void beginCompletion()
	{
		// Completion must be executed on a namer
		if (m_selection.node instanceof ConstantOperator &&
			((ConstantOperator)(m_selection.node)).getValue().isNamer())
		{
			// New state
			m_state = m_stateCompletion;

			// Create a new completion with namer as prefix
			m_completion = new Completion(((ConstantOperator)(m_selection.node)).getValue().getValue());

			// Execute one step of completion process
			continueCompletion();
		}
	}

	// Continue completion
	protected void continueCompletion()
	{
		String suffix = m_completion.nextSuffix();

		// Break process if no suffix
		if (suffix == null)
		{
			m_state = m_stateExpression;
			m_completion = null;
		}

		// Else change value of namer
		else
			((ConstantOperator)(m_selection.node)).setValue(m_completion.getPrefix() + suffix);
	}

	// Mouse selection
	public void mouseSelect(int x, int y)
	{
		m_selection.mouseSelect(this, x, y);
		m_hasOnlySelectionChanged = true;
	}

	// Automaton state
	protected int m_state;
	protected final int m_stateExpression = 0;
	protected final int m_stateCompletion = 1;

	// Selection
	protected Selection m_selection, m_oldSelection;
	protected boolean m_hasOnlySelectionChanged;

	// Completion
	protected Completion m_completion;

	// Clipboard
	protected CollectorOperator m_clipboard;
}