//
// Graphical display methods of graphic boxes
//


import java.awt.Graphics;

import EditableString;
import GraphicBox;
import Symbol;

abstract class Display extends GraphicBox
{
	// Constructor
	Display(Operator node)
	{
		m_node = node;
	}

	// Draw
	public void draw(Graphics graphics)
	{
		Display display;
		for (int index=0; (display=m_node.getChildDisplay(index))!=null; ++index)
			display.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{
		x = baseX + relativeX;
		y = baseY + relativeY;
		
		Display display;
		for (int index=0; (display=m_node.getChildDisplay(index))!=null; ++index)
			display.validate(x, y);
	}

	// Calculate
	public abstract void calculate(LocalContext localContext);

	// Access to children boxes
	protected Operator m_node;
}

class FormulaDisplay extends Display
{
	// Constructor
	FormulaDisplay(Operator node, int baseX, int baseY)
	{
		super(node);
		x = baseX;
		y = baseY;
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display child = m_node.getChildDisplay(0);

		// Children dimensions
		child.calculate(localContext);

		// Children vertical relative coordinates
		child.relativeX = 0;

		// Children horizontal relative coordinates
		child.relativeY = 0;

		// Dimensions
		width = child.width;
		height = child.height;
		yarn = child.yarn;
	}
}

class TemplateDisplay extends Display
{
	// Constructor
	TemplateDisplay(Operator node)
	{
		super(node);
		m_symbol = new TemplateSymbol(m_node.getContext());
	}

	// Draw
	public void draw(Graphics graphics)
	{	
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{			
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		// Children dimensions
		m_symbol.calculate(localContext);

		// Children vertical relative coordinates
		m_symbol.relativeX = 0;

		// Children horizontal relative coordinates
		m_symbol.relativeY = 0;

		// Dimensions
		width = m_symbol.width;
		height = m_symbol.height + GraphicBox.deltaHeight;
		yarn = m_symbol.yarn;
	}

	// Associated symbol (template)
 	protected TemplateSymbol m_symbol;
}

class PiDisplay extends Display
{
	// Constructor
	PiDisplay(Operator node)
	{
		super(node);
		m_symbol = new PiSymbol(m_node.getContext());
	}

	// Draw
	public void draw(Graphics graphics)
	{	
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{			
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		// Children dimensions
		m_symbol.calculate(localContext);

		// Children vertical relative coordinates
		m_symbol.relativeX = 0;

		// Children horizontal relative coordinates
		m_symbol.relativeY = 0;

		// Dimensions
		width = m_symbol.width;
		height = m_symbol.height + GraphicBox.deltaHeight;
		yarn = m_symbol.yarn;
	}

	// Associated symbol (template)
 	protected PiSymbol m_symbol;
}

class ConstantDisplay extends Display
{
	// Constructor
	ConstantDisplay(Operator node, EditableString editableString)
	{
		super(node);
		m_symbol = new StringSymbol(m_node.getContext(), editableString);
	}

	// Draw
	public void draw(Graphics graphics)
	{
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{			
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		// Children dimensions
		m_symbol.calculate(localContext);

		// Children vertical relative coordinates
		m_symbol.relativeX = 1;

		// Children horizontal relative coordinates
		m_symbol.relativeY = 0;

		// Dimensions
		width = m_symbol.width + 1;
		height = m_symbol.height + GraphicBox.deltaHeight;
		yarn = m_symbol.yarn;
	}

	// Associated symbol (template)
 	protected StringSymbol m_symbol;
}

class BracketsDisplay extends Display
{
	// Constructor
	BracketsDisplay(Operator node)
	{
		super(node);
		m_symbol = new BracketsSymbol(m_node.getContext());
	}

	// Draw
	public void draw(Graphics graphics)
	{
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display child = m_node.getChildDisplay(0);

		// Children dimensions
		child.calculate(localContext);
		m_symbol.calculate(localContext, child);

		// Children vertical relative coordinates
		child.relativeX = 1 + m_symbol.bracketWidth + GraphicBox.deltaWidth;
		m_symbol.relativeX = 1;

   		// Children horizontal relative coordinates
		child.relativeY = 1;
		m_symbol.relativeY = 1;

		// Dimensions
		width = m_symbol.width + 2;
		height = m_symbol.height + 2;
		yarn = m_symbol.yarn + 1;
	}

	// Associated symbol
	protected BracketsSymbol m_symbol;
}

class LeftUnaryDisplay extends Display
{
	// Constructor
	LeftUnaryDisplay(Operator node, String string)
	{
		super(node);
		m_symbol = new StringSymbol(m_node.getContext(), new EditableString(string));
	}

	// Draw
	public void draw(Graphics graphics)
	{
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display child = m_node.getChildDisplay(0);

		// Children dimensions
		child.calculate(localContext);
		m_symbol.calculate(localContext);

		// Children vertical relative coordinates
		child.relativeX = m_symbol.width + GraphicBox.deltaWidth;
		m_symbol.relativeX = 0;

		// Children horizontal relative coordinates
		m_symbol.relativeY = 0;  // Arbitrary to solve equation of yarn
		child.relativeY = m_symbol.yarn - child.yarn;
		if (child.relativeY < 0)
		{
			m_symbol.relativeY -= child.relativeY;
			child.relativeY = 0;
		}

		// Dimensions
		width = child.width + m_symbol.width + GraphicBox.deltaWidth;
		height = Math.max(child.relativeY+child.height,
			              m_symbol.relativeY+m_symbol.height);
		yarn = m_symbol.relativeY + m_symbol.yarn;
	}

	// Associated symbol
	protected StringSymbol m_symbol;
}

class BinaryDisplay extends Display
{
	// Constructor
	BinaryDisplay(Operator node, String string)
	{
		super(node);
		m_symbol = new StringSymbol(m_node.getContext(), new EditableString(string));
	}

	// Draw
	public void draw(Graphics graphics)
	{
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display left = m_node.getChildDisplay(0);
		Display right = m_node.getChildDisplay(1);

		// Children dimensions
		left.calculate(localContext);
		m_symbol.calculate(localContext);
		right.calculate(localContext);

		// Children vertical relative coordinates
		left.relativeX = 0;
		m_symbol.relativeX = left.width + GraphicBox.deltaWidth;
		right.relativeX = left.width + m_symbol.width + 2 * GraphicBox.deltaWidth;

		// Children horizontal relative coordinates
		left.relativeY = 0;  // Arbitrary to solve equation of yarn
		m_symbol.relativeY = left.yarn - m_symbol.yarn;
		right.relativeY = left.yarn - right.yarn;
		int delta = Math.min(m_symbol.relativeY, right.relativeY);
		if (delta < 0)
		{
			left.relativeY -= delta;
			m_symbol.relativeY -= delta;
			right.relativeY -= delta;
		}

		// Dimensions
		width = left.width + m_symbol.width + right.width + 2 * GraphicBox.deltaWidth;
		height = Math.max(left.relativeY+left.height,
			              Math.max(m_symbol.relativeY+m_symbol.height,
						           right.relativeY+right.height));
		yarn = left.relativeY + left.yarn;
	}

	// Associated symbol
	protected StringSymbol m_symbol;
}

class ComposeDisplay extends Display
{
	// Constructor
	ComposeDisplay(Operator node)
	{
		super(node);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display left = m_node.getChildDisplay(0);
		Display right = m_node.getChildDisplay(1);

		// Children dimensions
		left.calculate(localContext);
		right.calculate(localContext);

		// Children vertical relative coordinates
		left.relativeX = 0;
		right.relativeX = left.width + GraphicBox.deltaWidth;

		// Children horizontal relative coordinates
		left.relativeY = 0;  // Arbitrary to solve equation of yarn
		right.relativeY = left.yarn - right.yarn;
		int delta = right.relativeY;
		if (delta < 0)
		{
			left.relativeY -= delta;
			right.relativeY -= delta;
		}

		// Dimensions
		width = left.width + right.width + GraphicBox.deltaWidth;
		height = Math.max(left.relativeY+left.height,
			              right.relativeY+right.height);
		yarn = left.relativeY + left.yarn;
	}
}

class SuperscriptDisplay extends Display
{
	// Constructor
	SuperscriptDisplay(Operator node)
	{
		super(node);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display left = m_node.getChildDisplay(0);
		Display right = m_node.getChildDisplay(1);

		// Children dimensions
		left.calculate(localContext);
		LocalContext newLocalContext;
		if (localContext.fontSize >= m_node.getContext().minimalFontSize)
		{
			newLocalContext = new LocalContext(localContext);
			newLocalContext.fontSize = (newLocalContext.fontSize * m_node.m_context.fontReduce) / 100;
		}
		else
			newLocalContext = localContext;
		right.calculate(newLocalContext);

		// Children vertical relative coordinates
		left.relativeX = 0;
		right.relativeX = left.width;

		// Children horizontal relative coordinates
		left.relativeY = right.height;
		right.relativeY = 0;

		// Dimensions
		width = left.width + right.width;
		height = left.height + right.height;
		yarn = left.relativeY + left.yarn;
	}
}

class SubscriptDisplay extends Display
{
	// Constructor
	SubscriptDisplay(Operator node)
	{
		super(node);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display left = m_node.getChildDisplay(0);
		Display right = m_node.getChildDisplay(1);

		// Children dimensions
		left.calculate(localContext);
		LocalContext newLocalContext;
		if (localContext.fontSize >= m_node.getContext().minimalFontSize)
		{
			newLocalContext = new LocalContext(localContext);
			newLocalContext.fontSize = (newLocalContext.fontSize * m_node.m_context.fontReduce) / 100;
		}
		else
			newLocalContext = localContext;
		right.calculate(newLocalContext);

		// Children vertical relative coordinates
		left.relativeX = 0;
		right.relativeX = left.width;

		// Children horizontal relative coordinates
		left.relativeY = 0;
		right.relativeY = left.height;

		// Dimensions
		width = left.width + right.width;
		height = left.height + right.height;
		yarn = left.relativeY + left.yarn;
	}
}

class FractionDisplay extends Display
{
	// Constructor
	FractionDisplay(Operator node)
	{
		super(node);
		m_symbol = new FractionSymbol(m_node.getContext());
	}

	// Draw
	public void draw(Graphics graphics)
	{
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display left = m_node.getChildDisplay(0);
		Display right = m_node.getChildDisplay(1);

		// Children dimensions
		left.calculate(localContext);
		right.calculate(localContext);
		m_symbol.calculate(localContext, left, right);

		// Children vertical relative coordinates
		left.relativeX = 1 + (m_symbol.width - left.width) / 2;
		right.relativeX = 1 + (m_symbol.width - right.width) / 2;
		m_symbol.relativeX = 1;

		// Children horizontal relative coordinates
		left.relativeY = 0;
		m_symbol.relativeY = left.height + GraphicBox.deltaHeight;
		right.relativeY = left.height + m_symbol.height + 2 * GraphicBox.deltaHeight;

		// Dimensions
		width = m_symbol.width + 2;
		height = left.height + m_symbol.height + right.height + 2 * GraphicBox.deltaHeight;
		yarn = m_symbol.relativeY + m_symbol.yarn;
	}

	// Associated symbol
	protected FractionSymbol m_symbol;
}

class SqrtDisplay extends Display
{
	// Constructor
	SqrtDisplay(Operator node)
	{
		super(node);
		m_symbol = new SqrtSymbol(m_node.getContext());
	}

	// Draw
	public void draw(Graphics graphics)
	{
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display child = m_node.getChildDisplay(0);

		// Children dimensions
		child.calculate(localContext);
		m_symbol.calculate(localContext, child);

		// Children vertical relative coordinates
		child.relativeX = 1 + m_symbol.preSlash + GraphicBox.deltaWidth;
		m_symbol.relativeX = 1;

   		// Children horizontal relative coordinates
		child.relativeY = 1 + m_symbol.height - child.height;
		m_symbol.relativeY = 1;

		// Dimensions
		width = m_symbol.width + 2;
		height = m_symbol.height + 1;
		yarn = m_symbol.yarn + 1;
	}

	// Associated symbol
	protected SqrtSymbol m_symbol;
}

class VectorDisplay extends Display
{
	// Constructor
	VectorDisplay(Operator node)
	{
		super(node);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display display;

		// Children dimensions
		for (int index=0; (display=m_node.getChildDisplay(index))!=null; ++index)
			display.calculate(localContext);

		// Children vertical relative coordinates
		// Find maximum width
		int maximumWidth = 0;
		for (int index=0; (display=m_node.getChildDisplay(index))!=null; ++index)
		{
			if (display.width > maximumWidth)
				maximumWidth = display.width;
		}

		for (int index=0; (display=m_node.getChildDisplay(index))!=null; ++index)
			display.relativeX = (maximumWidth - display.width) / 2;

   		// Children horizontal relative coordinates
		int accumulateY = 0;
		for (int index=0; (display=m_node.getChildDisplay(index))!=null; ++index)
		{
			display.relativeY = accumulateY;
			accumulateY += display.height + GraphicBox.deltaHeight;
		}

		// Dimensions
		width = maximumWidth;
		height = accumulateY - GraphicBox.deltaHeight;
		yarn = height / 2;
	}
}

class MatrixDisplay extends Display
{
	// Constructor
	MatrixDisplay(Operator node)
	{
		super(node);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display display;
		Operator_matrix matrix = (Operator_matrix)m_node;

		// Children dimensions
		for (int index=0; (display=m_node.getChildDisplay(index))!=null; ++index)
			display.calculate(localContext);

		// Children vertical relative coordinates
		int accumulateX = 0;
		for (int column=0; column<matrix.getColumns(); ++column)
		{
			// Find maximum width on column
			int maximumWidth = 0;
			for (int row=0; (display=matrix.getChildDisplay(row, column))!=null; ++row)
			{
				if (display.width > maximumWidth)
					maximumWidth = display.width;
			}
			// Set vertical x coordinate on column
			for (int row=0; (display=matrix.getChildDisplay(row, column))!=null; ++row)
				display.relativeX = accumulateX + (maximumWidth - display.width) / 2;
			// Accumulate
			accumulateX += maximumWidth + 2 * GraphicBox.deltaWidth;
		}

   		// Children horizontal relative coordinates
		int accumulateY = 0;
		for (int row=0; row<matrix.getRows(); ++row)
		{
			// Find maximum height on row
			int maximumHeight = 0;
			for (int column=0; (display=matrix.getChildDisplay(row, column))!=null; ++column)
			{
				if (display.height > maximumHeight)
					maximumHeight = display.height;
			}
			// Set horizontal y coordinate on row
			// (1) Set relatively to accumulateY
			int delta = 0;
			for (int column=0; (display=matrix.getChildDisplay(row, column))!=null; ++column)
			{
				display.relativeY = accumulateY - display.yarn;
				if (accumulateY - display.relativeY > delta)
					delta = accumulateY - display.relativeY;
				
			}
			// (2) Correct to the new base: accumulateY + delta
			for (int column=0; (display=matrix.getChildDisplay(row, column))!=null; ++column)
				display.relativeY += delta;
			// Accumulate
			accumulateY += maximumHeight + 2 * GraphicBox.deltaHeight;
		}

		// Dimensions
		width = accumulateX - 2 * GraphicBox.deltaWidth;
		height = accumulateY - 2 * GraphicBox.deltaHeight;
		yarn = height / 2;
	}
}

class SigmaDisplay extends Display
{
	// Constructor
	SigmaDisplay(Operator node)
	{
		super(node);
		m_symbol = new SigmaSymbol(m_node.getContext());
	}

	// Draw
	public void draw(Graphics graphics)
	{
		super.draw(graphics);
		m_symbol.draw(graphics);
	}

	// Validate
	public void validate(int baseX, int baseY)
	{
		super.validate(baseX, baseY);
		m_symbol.validate(x, y);
	}

	// Calculate
	public void calculate(LocalContext localContext)
	{
		Display child = m_node.getChildDisplay(0);
		Display childDown = m_node.getChildDisplay(1);
		Display childUp = m_node.getChildDisplay(2);

		// Children dimensions
		child.calculate(localContext);
		if (child instanceof SigmaDisplay)
			m_symbol.calculate(localContext, ((SigmaDisplay)child).m_symbol);
		else
			m_symbol.calculate(localContext, child);
		LocalContext newLocalContext;
		if (localContext.fontSize >= m_node.getContext().minimalFontSize)
		{
			newLocalContext = new LocalContext(localContext);
			newLocalContext.fontSize = (newLocalContext.fontSize * m_node.m_context.fontReduce) / 100;
		}
		else
			newLocalContext = localContext;
		childDown.calculate(newLocalContext);
		childUp.calculate(newLocalContext);

		// Children vertical relative coordinates
		int maximumWidth = Math.max(childDown.width, Math.max(childUp.width, m_symbol.width));
		m_symbol.relativeX = 1 + (maximumWidth - m_symbol.width) / 2;
		childDown.relativeX = 1 + (maximumWidth - childDown.width) / 2;
		childUp.relativeX = 1 + (maximumWidth - childUp.width) / 2;
		child.relativeX = 1 + maximumWidth + GraphicBox.deltaWidth;

   		// Children horizontal relative coordinates
		childUp.relativeY = 1;
		m_symbol.relativeY = childUp.relativeY + childUp.height + GraphicBox.deltaHeight;
		childDown.relativeY = m_symbol.relativeY + m_symbol.height + GraphicBox.deltaHeight;
		child.relativeY = m_symbol.relativeY + m_symbol.yarn - child.yarn;

		// Dimensions
		width = maximumWidth + GraphicBox.deltaWidth + child.width + 2;
		height = Math.max(2+2*GraphicBox.deltaHeight+childUp.height+m_symbol.height+childDown.height,
		                  2+2*GraphicBox.deltaHeight+childUp.height+child.height+childDown.height);
		yarn = m_symbol.relativeY + m_symbol.yarn;
	}

	// Associated symbol
	protected SigmaSymbol m_symbol;
}
