//
// Algebraic rules for operators
//

import java.util.Stack;

import Context;
import EditableString;
import Operator;
import Topology;

// an inserand is inserted at an inseror (or inseror parents in cut)

class CollectorOperator extends Operator
{
	// Constructor
	CollectorOperator(Context context)
	{
		super(context, 1, 0, Operator.maximumPriority);
		m_display = null;  // A collector is not an algebraic operator (the only)

		// Create child
		m_children[0] = new TemplateOperator(m_context);
		link(this, m_children[0], 0);
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		return inseror;  // Never inserted
	}
}

class TemplateOperator extends Operator
{
	// Constructor
	TemplateOperator(Context context)
	{
		super(context, 0, Operator.maximumPriority, Operator.maximumPriority);
		m_display = new TemplateDisplay(this);
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		Cut cut = findCut(inseror);
		flopAsIs(cut.node);
		remove(cut.node);
		return this;
	}

	// Delete
	public Operator delete(int location)
	{
		return m_parent.deleteChild(m_rank);
	}
}

abstract class PreConstantOperator extends Operator
{
	// Constructor
	PreConstantOperator(Context context)
	{
		super(context, 0, Operator.maximumPriority, Operator.maximumPriority);
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		Cut cut = findCut(inseror);
		flopAsIs(cut.node);
		remove(cut.node);
		return this;
	}
}

class ConstantOperator extends Operator
{
	// Constructor
	ConstantOperator(Context context, String string)
	{
		super(context, 0, Operator.maximumPriority, Operator.maximumPriority);
		m_value = new EditableString(string);
		m_display = new ConstantDisplay(this, m_value);
	}

	// Constructor
	ConstantOperator(Context context, char character)
	{
		super(context, 0, Operator.maximumPriority, Operator.maximumPriority);
		m_value = new EditableString(character);
		m_display = new ConstantDisplay(this, m_value);
	}

	// Value accessor
	public EditableString getValue()
	{
		return m_value;
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		Cut cut = findCut(inseror);
		flopAsIs(cut.node);
		remove(cut.node);
		return this;
	}
	
	// Delete
	public Operator delete(int location)
	{
		if (location!=Topology.RIGHT || m_value.delete())
			return this;
		else
			return super.delete(location);
	}

	// Extend value identifier
	public Operator extend(char character)
	{
		m_value.insert(character);
		return this;
	}

	// Extend value identifier
	public Operator extend(String string)
	{
		if (string == null)
			return this;

		m_value.insert(string);
		return this;
	}

	// Set value identifier
	public void setValue(String value)
	{
		m_value.setValue(value);
	}

	// Associated value
	protected EditableString m_value;
}

abstract class UnitaryOperator extends Operator
{
	// Constructor
	UnitaryOperator(Context context, int arity)
	{
		super(context, arity,  Operator.maximumPriority, Operator.maximumPriority);
		for (int index=0; index<arity; ++index)
			m_breakers[index] = BREAK_CUT;
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		if (m_children != null)
		{
			// Absorb cut as child 0, and create other children

			Cut cut = findCut(inseror);

			flopAsIs(cut.node);
			link(this, cut.node, 0);

			for (int index=1; index<m_children.length; ++index)
			{
				TemplateOperator templateOperator = new TemplateOperator(m_context);
				link(this, templateOperator, index);
			}

			return cut.node;
		}

		// Should never be here
		return null;
	}

	// Delete node operation
	public Operator delete(int location)
	{
		if (m_children.length != 1)
			super.delete(location);

		if (location != Topology.RIGHT)
			return this;

		remove(this);
		link(m_parent, m_children[0], m_rank);
		return m_children[0];
	}
}

class BracketsOperator extends UnitaryOperator
{
	// Constructor
	BracketsOperator(Context context)
	{
		super(context, 1);
		m_display = new BracketsDisplay(this);
		m_topology.setEntering(0, 0, 0, 0);
	}
}

class LeftUnaryOperator extends Operator
{
	// Constructor
	LeftUnaryOperator(Context context, int asInserorPriority, int asInserandPriority)
	{
		super(context, 1, asInserorPriority, asInserandPriority);
		m_topology.setEntering(0, 0, 0, 0);
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		Cut cut = findCut(inseror);
		
		if (cut.child != null)
		{
			link(cut.node, this, cut.child.m_rank);
			link(this, cut.child, 0);
		}
		else
		{
			link(cut.node.m_parent, this, cut.node.m_rank);
			link(this, cut.node, 0);
		}

		return m_children[0];
	}

	// Reconstruct "this" for non linear insert
	public Operator reconstructByInjection(Stack stack, Operator destination, Operator inserand, int location, Operator continuation)
	{
		Operator linear = destination;
		Operator right = m_children[0];
		Operator child = (Operator)(stack.pop());

		// Reconstruct right operand
		linear = flop(linear);
		linear = right.reconstruct(stack, linear, inserand, location, continuation);

		return linear;
	}

	// Delete node operation
	public Operator delete(int location)
	{
		if (location != Topology.RIGHT)
			return this;

		remove(this);
		link(m_parent, m_children[0], m_rank);
		return m_children[0];
	}

	// Delete child
	public Operator deleteChild(int childRank)
	{
		Operator child = m_children[childRank];

		remove(this);
		link(m_parent, child, m_rank);
		
		return child;
	}
}

abstract class BinaryOperator extends Operator
{
	// Constructor
	BinaryOperator(Context context, int asInserorPriority, int asInserandPriority)
	{
		super(context, 2, asInserorPriority, asInserandPriority);
		Moving[] movings = new Moving[2];
		movings[0] = new Moving(0, Topology.RIGHT, 1);
		movings[1] = new Moving(1, Topology.LEFT, 0);
		m_topology.setMovings(movings);
		m_topology.setEntering(1, 0, 0, 0);
	}

	// Reconstruct "this" for non linear insert
	public Operator reconstructByInjection(Stack stack, Operator destination, Operator inserand, int location, Operator continuation)
	{
		Operator linear = destination;
		Operator left = m_children[0];
		Operator right = m_children[1];
		Operator child = (Operator)(stack.pop());

		if (child == left)
		{
			// Reconstruct left operand
			linear = left.reconstruct(stack, linear, inserand, location, continuation);
			linear = flop(linear);
			linear = right.flopAsIs(linear);
		}
		else
		{
			// Reconstruct right operand
			linear = left.flopAsIs(linear);
			linear = flop(linear);
			linear = right.reconstruct(stack, linear, inserand, location, continuation);
		}

		return linear;
	}

	// Delete child
	public Operator deleteChild(int childRank)
	{
		Operator child = m_children[(childRank+1)%2];

		remove(m_children[childRank]);
		remove(this);
		link(m_parent, child, m_rank);
		
		return child;
	}
}

class LeftBinaryOperator extends BinaryOperator
{
	// Constructor
	LeftBinaryOperator(Context context, int asInserorPriority, int asInserandPriority)
	{
		super(context, asInserorPriority, asInserandPriority);
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		Cut cut = findCut(inseror);
		TemplateOperator templateOperator = new TemplateOperator(m_context);
		
		if (m_asInserandPriority == cut.node.m_asInserorPriority)  // Use left associativity
		{
			link(cut.node.m_parent, this, cut.node.m_rank);
			link(this, cut.node, 0);
			link(this, templateOperator, 1);
		}
		else if (cut.child != null)
		{
			link(cut.node, this, cut.child.m_rank);
			link(this, cut.child, 0);
			link(this, templateOperator, 1);
		}
		else
		{
			link(cut.node.m_parent, this, cut.node.m_rank);
			link(this, cut.node, 0);
			link(this, templateOperator, 1);
		}

		return templateOperator;
	}
}

class RightBinaryOperator extends BinaryOperator
{
	// Constructor
	RightBinaryOperator(Context context, int asInserorPriority, int asInserandPriority)
	{
		super(context, asInserorPriority, asInserandPriority);
	}

	// Insert "this" at "operator"
	public Operator flop(Operator inseror)
	{
		Cut cut = findCut(inseror);
		TemplateOperator templateOperator = new TemplateOperator(m_context);
		
		if (m_asInserandPriority == cut.node.m_asInserorPriority)  // Use right associativity
		{
			Operator node = cut.node.m_children[1];
			while (node!=null && m_asInserandPriority==node.m_asInserorPriority)
				node = node.m_children[1];
			Operator parent = node.m_parent;
			link(parent, this, 1);
			link(this, node, 0);
			link(this, templateOperator, 1);
		}
		else if (cut.child != null)  // No associativity but the same code
		{
			link(cut.node, this, cut.child.m_rank);
			link(this, cut.child, 0);
			link(this, templateOperator, 1);
		}
		else
		{
			link(cut.node.m_parent, this, cut.node.m_rank);
			link(this, cut.node, 0);
			link(this, templateOperator, 1);
		}

		return templateOperator;
	}
}