import java.io.*;
import java.net.*;
import java.util.*;

/**
 * <B>FMserverConnection class</B>.<BR><BR>
 * Handles a connection with a client answering to all the
 * requests.
 */
class FMServerConnection extends Thread {
  /**
   * Requests and replies codes 
   */
  public static final int FAILURE = 0x00;
  public static final int SUCCESS = 0x01;
  public static final int REGISTER = 0x02;
  public static final int LOGIN = 0x03;
  public static final int LOGOUT = 0x04;
  public static final int QUERY = 0x05;
  public static final int ONLINE = 0x06;
  public static final int MESSAGE = 0x07;
  public static final int GETMESSAGES = 0x08;
  
  public static final int QUERY_BY_ID = 0x00;
  public static final int QUERY_BY_NICKNAME = 0x01;
  public static final int QUERY_BY_REALNAME = 0x02;
  public static final int QUERY_BY_EMAIL = 0x03;

  Socket connectionSocket;
  DataInputStream is;
  DataOutputStream os;
  FMServer server;
  FMUserDatabase userDatabase;
  
  /**
   * Public constructor.
   * Initializes the connection on the specified socket
   *
   * @param server The server which opened the connection.
   * @param userDatabase The user database to be used for the requests.
   * @param connectionSocket The socket used for this connection.
   */
  public FMServerConnection(FMServer server, FMUserDatabase userDatabase, Socket connectionSocket) throws IOException {
    this.server = server;
    this.userDatabase = userDatabase;
    this.connectionSocket = connectionSocket;
    
    is = new DataInputStream(connectionSocket.getInputStream());
    os = new DataOutputStream(connectionSocket.getOutputStream());
  };
  
  /**
   * Main execution thread.
   * Read a request code and process it.
   */
  public void run() {
    FMUser user;
    boolean done;
    
    done = false;
    while(!done) {
      int request;
      
      try {
	request = is.readInt();	/* Get the request ID */
	switch(request) {
	 
	case REGISTER: handleRegisterRequest();
	  break;
	  
	case LOGIN: handleLoginRequest();
	  break;
      
	case LOGOUT: handleLogoutRequest();
	  break;
	  
	case QUERY: handleQueryRequest();
	  break;

	case ONLINE: handleOnlineRequest();
	  break;
	case MESSAGE: handleMessageRequest();
	  break;
	case GETMESSAGES: handleGetMessagesRequest();
	  break;
     	};
	
      }
      catch(EOFException exception) {
	done = true;
      }
      catch(IOException exception) {
	System.err.println("Error reading data");
	done = true;
      };
    };
    
    try {
      connectionSocket.close();
    }
    catch(IOException exception) {};
  };

  private void handleRegisterRequest() throws IOException {
    FMUser user;

    user = new FMUser();
    user.read(is);
    
    try {
      userDatabase.addUser(user);
    }
    catch(IOException exception) {
      os.writeInt(FAILURE);
      return;
    };

    os.writeInt(SUCCESS);
    os.writeInt(user.getID());
  };

  /**
   * Login
   *
   * Request
   *  ID (int)
   *  Password (String)
   *  Host port (int)
   *
   * Reply
   *  FAILURE
   *  SUCCESS + User data
   */
  private void handleLoginRequest() throws IOException {
    FMUser user;
    int userID;
    String password;
    int port;

    userID = is.readInt();
    password = is.readUTF();
    port = is.readInt();
    
    try {
      user = userDatabase.getUser(userID);
    }
    catch(IOException exception) {
      os.writeInt(FAILURE);
      return;
    };

    if(password.compareTo(user.getPassword()) != 0) {
      os.writeInt(FAILURE);
      return;
    };

    user.setHostAddress(connectionSocket.getInetAddress());
    user.setHostPort(port);

    server.addUser(user);

    os.writeInt(SUCCESS);
    user.write(os);
  };

  /**
   * Logout
   *
   * Request
   *  ID (int)
   *  Password (String)
   *
   * Reply
   *  FAILURE
   *  SUCCESS + User data
   */
  private void handleLogoutRequest() throws IOException {
    FMUser user;
    int userID;
    String password;
    int port;

    userID = is.readInt();
    password = is.readUTF();
     
    try {
      user = userDatabase.getUser(userID);
    }
    catch(IOException exception) {
      os.writeInt(FAILURE);
      return;
    };

    if(password.compareTo(user.getPassword()) != 0) {
      os.writeInt(FAILURE);
      return;
    };

    server.removeUser(user);

    os.writeInt(SUCCESS);
  };

  
  /**
   * Query
   *
   * Request
   *  Type (int)
   *  Data (String)
   *
   * Reply
   *  SUCCESS + Number of matchings (int) + User data * n
   */  
  private void handleQueryRequest() throws IOException{
    int queryType;
    String queryData;
    Vector matchings;
    int userID;
    Enumeration e;

    queryType = is.readInt();
    queryData = is.readUTF();
    matchings = null;

    switch(queryType) {
    case QUERY_BY_ID: 
      try {
	userID = Integer.parseInt(queryData);
      }
      catch(NumberFormatException exception) {
	userID = 0;
      };

      matchings = userDatabase.queryByID(userID);
      break;
      
    case QUERY_BY_NICKNAME: matchings = userDatabase.queryByNickName(queryData);
      break;

    case QUERY_BY_REALNAME: matchings = userDatabase.queryByRealName(queryData);
      break;

    case QUERY_BY_EMAIL: matchings = userDatabase.queryByEMail(queryData);
      break;
    };

    os.writeInt(SUCCESS);
    os.writeInt(matchings.size());
    e = matchings.elements();
    while(e.hasMoreElements()) {
      FMUser currentUser;

      currentUser = (FMUser)e.nextElement();
      currentUser.write(os);
    };
  };

  /**
   * Online
   *
   * Request
   *   User ID (int)
   *
   * Reply
   *  FAILURE
   *  SUCCESS + User data
   */
  private void handleOnlineRequest() throws IOException {
    int userID;
    Vector connectedUsers;
    Enumeration e;

    userID = is.readInt();
    connectedUsers = server.getUsers();
    e = connectedUsers.elements();
    while(e.hasMoreElements()) {
      FMUser user;
      
      user = (FMUser)e.nextElement();
      if(userID == user.getID()) {
	os.writeInt(SUCCESS);
	user.writeAll(os);
	return;
      };
    };

    os.writeInt(FAILURE);
  };

  private void handleMessageRequest() throws IOException {
    DataOutputStream fos;
    FMOffLineMessage offLineMessage;
    
    offLineMessage = new FMOffLineMessage();
    offLineMessage.read(is);
    os.writeInt(SUCCESS);

    fos = new DataOutputStream(new FileOutputStream("Users/" + offLineMessage.to.getID() + ".messages", true));

    offLineMessage.write(fos);
    fos.close();
  };

  private void handleGetMessagesRequest() throws IOException {
    FMUser user;
    int userID;
    String password;
    int port;
    File messageFile;
    DataInputStream fis;
    Vector messages;
    Enumeration e;

    userID = is.readInt();
    password = is.readUTF();
     
    try {
      user = userDatabase.getUser(userID);
    }
    catch(IOException exception) {
      os.writeInt(FAILURE);
      return;
    };

    if(password.compareTo(user.getPassword()) != 0) {
      os.writeInt(FAILURE);
      return;
    };

    messages = new Vector(10, 5);

    try {
      messageFile = new File("Users/" + userID + ".messages");
      fis = new DataInputStream(new FileInputStream(messageFile));
      while(fis.available() > 0) {
	FMOffLineMessage message;

	message = new FMOffLineMessage();
	message.read(fis);
	messages.addElement(message);
      };

      fis.close();
      messageFile.delete();
    }
    catch(FileNotFoundException exception) {
    }
    catch(IOException exception) {
    };

    os.writeInt(SUCCESS);
    os.writeInt(messages.size());
    e = messages.elements();
    while(e.hasMoreElements()) {
      FMOffLineMessage message;

      message = (FMOffLineMessage)e.nextElement();
      message.write(os);
    };
  };
    

    

    
};
































