#! /bin/csh
#
# $Id: jtags.csh 1.2 1997/08/26 09:08:21 kinnucan Exp $
#
# Usage:     jtags [-h | srcdir]
#
# Abstract:  This Bourne shell script produces an Emacs
#            tags file for Java source code. The tags file
#            contains tags for public classes, interfaces,
#            and methods.
#
# Options:   -h  Print usage.
#
#            -srcdir  Path of top-level directory containing 
#                     Java source(*.java) files to be tagged.
#                     If omitted, this script tags files in
#                     the working directory and its subdirectories.
#
# By:        Paul Kinnucan 
#            The MathWorks, Inc.
#	     paulk@mathworks.com
#
# Thanks:    David Lim (david@ndc.com) and Michael Mirman
#	     (mmirman@mathworks.com) helped me greatly with
#            this script.
#
# History:
#
# $Log: jtags.csh $
# Revision 1.2  1997/08/26 09:08:21  kinnucan
# 1. Exclude RCS (version control) directories from tags search.
# 2. Added pattern for recognizing abstract classes.
#
#

set etags_dir = ""
set java_dir = .

# Process command line parameters.
foreach arg ($*)
    switch ($arg) 
	case -h:
	case -help:
        case -usage:
	    set help = 1
	    breaksw
	case -e*:
	    set etags_dir = `expr substr $arg 3 200`
	    breaksw
	case -j*:
	    set java_dir = `expr substr $arg 3 200`
	    breaksw
	default:
	    set invalid_arg = 1
	    breaksw
    endsw
end

# Print help.
if ( $1 == "-h" || $1 == "-help" || $1 == "-usage" ) then
    echo ""
    echo "usage: jtags [-h | srcdir]"
    echo ""
    echo "srcdir   Path of the Java source directory."
    echo ""
    echo "This command tags all public classes, interfaces,"
    echo "methods, and variables in the specified Java source"
    echo "hierarchy. If you do not specify a source directory," 
    echo "jtags searches the current directory and its"
    echo "subdirectories."
    echo ""
    exit 1
endif

# Ensure that etags directory exists.
if ( $etags_dir != "" ) then
    if (! -d $etags_dir ) then
	echo "etags directory $etags_dir not found."
	exit 1
    endif
endif

# If not the help option, assume parameter is the source path.
if ( $1 == "" ) then
    set java_dir = "."
else
    set java_dir = $1
endif

# Ensure that Java source directory exists.
if (! -e $java_dir ) then
    echo "Java source directory $java_dir not found."
    exit 1
endif
    
# Regular expression for tagging Java class types
set re_class1 = --regex='/[ \t]*\<public\>.*\<class\>[ \t]+\([a-zA-Z0-9_.]+\)[ \t]+.*/\1/'
set re_class2 = --regex='/[ \t]*\<abstract\>.*\<public\>.*\<class\>[ \t]+\([a-zA-Z0-9_.]+\)[ \t]+.*/\1/'
set re_class3 = --regex='/[ \t]*\<public\>.*\<abstract\>.*\<class\>[ \t]+\([a-zA-Z0-9_.]+\)[ \t]+.*/\1/'

# Regular expression for tagging Java interface types
set re_interface = --regex='/[ \t]*\<public\>.*\<interface\>[ \t]+\([a-zA-Z0-9_.]+\)[ \t]+.*/\1/'

# Tag Java methods
set re_method = --regex='/.*\<public\>[^=]*[ \t]+\([a-zA-Z0-9_]+\)[ \t]*(.*)\([ \t]*{.*\|[ \t]*\)/\1/'

# Tags Java variable declarations with user-defined types.
# Note: we assume user-defined types begin with a capital letter.
set re_variable1 = --regex='/.*\<[A-Z]+[A-Za-z0-9_.]+\>\(\[\][ \t]+\|[ \t]+\)\(\<[A-Za-z0-9_.]+\>\)\(\[\][ \t]*\|[ \t]*\)\(;\|=\).*/\2/'

# Regular expression for tagging Java variable declarations with built-in types.
set re_variable2 = --regex='/.*\(\<boolean\>\|\<char\>\|\<byte\>\|\<short\>\|\<int\>\|\<long\>\|\<float\>\|\<double\>\|\<float\>\)\(\[\][ \t]+\|[ \t]+\)\(\<[A-Za-z0-9_.]+\>\)\(\[\][ \t]*\|[ \t]*\)\(;\|=\).*/\3/'

# Delete the old TAGS file.
# Note: the old file must be deleted because we have to run
# etags in append mode. If we don't remove the old file, 
# etags will append everything to the old file.

if ( -e $java_dir/TAGS ) then
    rm $java_dir/TAGS
    echo "Removed old TAGS file."
endif

# Use find to recurse through the source hierarchy, 
# finding every java source file.
# Use xargs to apply etags to the source files.
# Note that xargs may invoke etags multiple
# times, depending on how many files it can batch
# per invocation. That is why we run etags in
# append (-a) mode.
/bin/find ${java_dir} \( -name RCS -prune \) -o \( -name '*.java' -print \) |\
xargs ${etags_dir}etags  -a -o ${java_dir}/TAGS \
"$re_class" "$re_method" "$re_interface"  \
"$re_variable1" "$re_variable2"



 