;; jde-run.el --- runs the Java app in the current buffer.
;; $Revision: 1.11 $ $Date: 1997/10/05 21:21:59 $

;; Author: Paul Kinnucan <paulk@mathworks.com>
;; Maintainer: Paul Kinnucan
;; Version 1.9
;; Keywords: tools, processes

;; Copyright (C) 1997 Free Software Foundation, Inc.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;; Change History
;; $Log: jde-run.el $
;; Revision 1.11  1997/10/05 21:21:59  kinnucan
;; Unquoted classpath as quotes are only necessary for compilation (because
;; the JDE uses a shell to run the compiler).
;;
;; Revision 1.10  1997/10/05 17:15:44  kinnucan
;; Added the function jde-run-set-app-args, which allows you to
;; specify command line arguments for the application you are running.
;;
;; Also, changed the value of jde-run-args from a string to a list.
;;
;; Revision 1.9  1997/09/16 02:37:16  kinnucan
;; Changed w32-start-process-show-window to win32-start-process-show-window
;;
;; Revision 1.8  1997/09/04 03:54:34  kinnucan
;; Added jde-run-applet command, which runs a Java applet.
;;
;; Revision 1.7  1997/08/29 03:19:04  kinnucan
;; Fixed bug in save-w32-show-window.
;;
;; Revision 1.6  1997/08/26 08:46:41  kinnucan
;; Tweaked version number.
;;
;; Revision 1.5  1997/08/26 08:33:16  kinnucan
;; Deleted superfluous comments.
;;
;; Revision 1.4  1997/08/26 08:31:36  kinnucan
;; 1. Ported jde-run onto comint mode.
;;
;;    This allows you to interact with a Java application in the
;;    run buffer, if the application accepts command line input.
;;    You can use the comint history features to facilitate interaction
;;    with such an application.
;;
;; 2. Added the jde-run-set-java-vm and jde-run-set-java-vm-w
;;    commands, which let you specify the Java interpreter to use to
;;    run on non-Windows and Windows platforms, respectively.
;;
;;    Note that you must use javaw on Windows platforms to avoid
;;    opening a superfluous command shell window.
;;
;; 3. Added the jde-run-set-args command and associated jde-run-args
;;    variable, which let you specify Java interpreter options via
;;    command-line arguments.
;;
;;    jde-run passes the value of jde-classpath (defined in jde.el
;;    and set via the jde-set-classpath command) and jde-run-args
;;    to the Java interpreter.
;;
;;   This means that you can use a common classpath definition for
;;   compiling and running applications, while passing other
;;   runtime arguments via jde-run-set-args.
;;
;; Revision 1.3  1997/07/05 04:20:44  kinnucan
;; Modified jde-run command to derive the class name from the name of the file in
;; the current buffer rather than the buffer name. This avoids an incorrect derivation
;; when more than one buffer is open on the same source file.
;;
;; Revision 1.2  1997/06/29 08:23:21  kinnucan
;; 1. Added jde-run-set-app function, which lets you specify the application
;;    class to run.
;;
;; 2. Updated jde-run to run either the app specified by jde-run-set-app or
;;    the class whose source is in the current buffer. In the latter case,
;;    jde-run extracts the package of the app class from the source buffer.
;;
;; Revision 1.1  1997/06/18 17:23:28  paulk
;; Initial revision
;;

(defvar jde-run-mode-hook nil
  "*List of hook functions run by `jde-run-mode' (see `run-hooks').")

(defvar jde-run-application-class nil
  "String specifying the name of the Java class to run. Note that this
class must have a static public main method.")

(defvar jde-run-java-vm "java"
  "Name of the Java interpreter used to run Java applications on non-Windows platforms. Defaults to java.")

(defvar jde-run-java-vm-w "javaw"
  "Name of the Java interpreter used to run Java applications on Windows platforms. Defaults to javaw.")

(defvar jde-run-args nil
  "List of arguments (except -classpath) to be passed to the Java vm")

(defvar jde-run-app-args nil
  "List of arguments to be passed to the Java application class.")

(defvar jde-run-applet-viewer "appletviewer"
  "Name of applet viewer. Devault is appletviewer.")

(defvar jde-run-applet-doc "index.html"
  "String specifying name of document containing applet to be viewed. If nil,
JDE assumes that the document name is APPLET.html, where APPLET is the name
of the applet to be viewed. The default value is index.html.")

;;(defun bashify-command (cmd)
;;  "If bash is the current shell, returns CMD enclosed in quotes.
;;This satisfies bash requirement that the -c command line option
;;be enclosed in qotes."
;;  (if (and (eq system-type 'windows-nt)
;;	   (or (string-match "bash" shell-file-name)
;;	       (string-match "bash" (getenv "SHELL"))))	  
;;      (concat "\"" cmd "\"")
;;    cmd))

(defun jde-run-set-vm (vm)
  "Specifies the Java interpreter used to run Java applications
on non-Windows platforms . The default is java."
  (interactive
   "sEnter name of Java interpreter: ")
  (setq jde-run-java-vm vm))

(defun jde-run-set-vm-w (vm)
  "Specifies the Java interpreter used to run Java applications
on Windows platforms . The default is javaw."
  (interactive
   "sEnter name of Java interpreter: ")
  (setq jde-run-java-vm-w vm))

(defun jde-run-set-app (app)
  "Specify the name of the application class to run."
  (interactive 
   "sEnter application class: ")
  (setq jde-run-application-class app))

(defun jde-run-set-args (args)
  "Specify the arguments (except -classpath) to be passed to the Java vm."
  (interactive 
   "sEnter arguments: ")
  (setq jde-run-args (jde-run-parse-args args)))

(defun jde-run-make-vm-arg-string ()
 (let ((classpath-arg (jde-get-classpath-arg))
       (other-args (jde-run-make-arg-string jde-run-args)))
   (if classpath-arg
       (setq classpath-arg (concat classpath-arg " ")))
   (if other-args
       (setq other-args (concat other-args " ")))
   (concat classpath-arg other-args)))

(defun jde-run-set-app-args (args)
  "Specify the arguments to be passed to the Java application class."
  (interactive 
   "sEnter arguments: ")
  (setq jde-run-app-args (jde-run-parse-args args)))

(defun jde-run-set-applet-viewer (viewer)
  "Sets the viewer to be used to view an applet. The default is 
appletviewer."
  (interactive
   "sEnter viewer name: ")
  (setq jde-run-applet-viewer viewer))

(defun jde-run-set-applet-doc (doc)
  "Specify the doc to be used to view an applet. If nil, JDE
assumes that the doc is named APPLET.html, where APPLET is the 
root name of the Java source file in the current buffer. The default
is index.html."
  (interactive
   "sEnter applet doc name: ")
  (if (string= doc "")
      (setq jde-run-applet-doc nil)
    (setq jde-run-applet-doc doc)))

(defun jde-run()
  "Runs the Java program named by jde-run-application-class in
a buffer, piping output from the program to the buffer and 
input from the buffer to the program."
  (interactive)
  (let ((app-class jde-run-application-class))
    (if (not app-class)
	(setq app-class
	      (concat (jde-db-get-package)
		      (file-name-sans-extension 
		       (file-name-nondirectory (buffer-file-name))))))
    (jde-run-internal app-class)))

(defmacro save-w32-show-window (&rest body)
"Saves the value of the w32-start-process-show-window variable
before evaluating body and restores the value afterwards."
  `(if (and 
	(eq system-type 'windows-nt)
	(boundp 'win32-start-process-show-window))
       (let ((save win32-start-process-show-window))
	 (setq win32-start-process-show-window t)
	 ,@body
	 (setq win32-start-process-show-window save))
    ,@body))

(defun jde-run-unquote (string)
  (if (eq (aref string 0) ?\")
      (substring string 1 (- (length string) 1))
    string))

(defun jde-run-internal(app-class)
  (let ((run-buf-name (concat "*" app-class "*")))
    (if (not (comint-check-proc run-buf-name))
	(let* ((run-buffer (get-buffer-create run-buf-name))
	       (win32-p (eq system-type 'windows-nt))
	       (prog (if (and
			  win32-p
			  (string= jde-run-java-vm "java"))
			 jde-run-java-vm-w
		       jde-run-java-vm))
	       (prog-args (append
			   (if (not
				(string= jde-classpath ""))
			       (list "-classpath" 
				     ;; Remove quotes required 
				     ;; for using classpath for
                                     ;; compilation, which runs in a shell.
				     (jde-run-unquote jde-classpath)))
			   jde-run-args
			   (list app-class)
			   jde-run-app-args))
	       (command-string (concat prog " " 
				       (jde-run-make-arg-string prog-args)
				       "\n\n")))
	  (save-excursion
	    (set-buffer run-buffer)
	    (erase-buffer)
	    (insert (concat "cd " default-directory "\n"))
	    (insert command-string)
	    (jde-run-mode))
	  (save-w32-show-window
	   (comint-exec run-buffer app-class prog nil prog-args))
	  (pop-to-buffer run-buffer))
      (message "An instance of %s is running." app-class)
      (pop-to-buffer run-buf-name))))


(defun jde-run-mode()
  (interactive)
  (comint-mode)
  (setq major-mode 'jde-run-mode)
  (run-hooks 'jde-run-mode-hook)
)

(defun jde-run-parse-args (s)
  (let ((n (string-match "[a-zA-Z0-9\\.:;\\_/-]+" s))
	(i 0)
	(tokens '()))
    (while n
      (setq tokens (append tokens (list (match-string 0 s))))
      (setq n (match-end 0))
      (setq n (string-match "[a-zA-Z0-9\\.:;\\_/-]+" s n)))
    tokens))

(defun jde-run-make-arg-string (args)
  (let ((str "")
	(n (length args))
	(i 0))
    (while (< i n)
      (if (not (string= str ""))
	  (setq str (concat str " ")))
      (setq str (concat str (nth i args)))
      (setq i (+ i 1)))
    str))
    
(defun jde-run-applet-exec (buffer name command startfile switches)
  "A version of comint-exec patched to start an applet viewer as
a command shell subprocess rather than as a subprocess of Emacs. This
is necessary to avoid displaying a DOS window when starting a viewer
under Windows."
  (save-excursion
    (set-buffer buffer)
    (let ((proc (get-buffer-process buffer)))	; Blast any old process.
      (if proc (delete-process proc)))
    ;; Crank up a new process
    (let ((proc (jde-run-applet-exec-1 name buffer command switches)))
      (set-process-filter proc 'comint-output-filter)
      (make-local-variable 'comint-ptyp)
      (setq comint-ptyp process-connection-type) ; T if pty, NIL if pipe.
      ;; Jump to the end, and set the process mark.
      (goto-char (point-max))
      (set-marker (process-mark proc) (point))
    (run-hooks 'comint-exec-hook)
    buffer)))

;; This auxiliary function cranks up the process for jde-run-applet-exec in
;; the appropriate environment.

(defun jde-run-applet-exec-1 (name buffer command switches)
  (let ((process-environment
	 (nconc
	  ;; If using termcap, we specify `emacs' as the terminal type
	  ;; because that lets us specify a width.
	  ;; If using terminfo, we specify `dumb' because that is
	  ;; a defined terminal type.  `emacs' is not a defined terminal type
	  ;; and there is no way for us to define it here.
	  ;; Some programs that use terminfo get very confused
	  ;; if TERM is not a valid terminal type.
	  (if (and (boundp 'system-uses-terminfo) system-uses-terminfo)
	      (list "TERM=dumb"
		    (format "COLUMNS=%d" (frame-width)))
	    (list "TERM=emacs"
		  (format "TERMCAP=emacs:co#%d:tc=unknown:" (frame-width))))
	  (if (getenv "EMACS") nil (list "EMACS=t"))
	  process-environment))
	(default-directory
	  (if (file-directory-p default-directory)
	      default-directory
	    "/")))
    (apply 'start-process-shell-command name buffer command switches)))

(defun jde-run-applet-internal (doc)
  (let* ((doc-name (file-name-sans-extension
			(file-name-nondirectory doc)))
	 (run-buf-name (concat "*" doc-name "*")))
    (if (not (comint-check-proc run-buf-name))
	(let* ((run-buffer (get-buffer-create run-buf-name))
	       (win32-p (eq system-type 'windows-nt))
	       (prog jde-run-applet-viewer)) 
	  (save-excursion
	    (set-buffer run-buffer)
	    (erase-buffer)
	    (insert (concat "cd " default-directory "\n"))
	    (insert (concat prog " "  doc "\n\n"))
	    (jde-run-mode))
	  (jde-run-applet-exec run-buffer doc-name prog nil (list doc))
	  (pop-to-buffer run-buffer))
      (message "An instance of %s is running." app-class)
      (pop-to-buffer run-buf-name))))

(defun jde-run-applet (&optional doc)
  "Run an applet, using the viewer specified by jde-run-applet-viewer
and the applet document specified by jde-run-applet-doc."
  (interactive
   "sEnter applet document name: ")
  (let ((applet-doc (if doc
			doc
		      (if jde-run-applet-doc
			  jde-run-applet-doc
			(concat 
			 (file-name-sans-extension buffer-file-name)
			 ".html")))))
	(jde-run-applet-internal 
	 (if (string-match "appletviewer" jde-run-applet-viewer)
	     applet-doc
	   (concat default-directory applet-doc)))))

(defun jde-run-menu-run-applet ()
  (interactive)
  (jde-run-applet jde-run-applet-doc))

(provide 'jde-run)

;;; jde-run.el ends here