/*
 * RemoteFileDialog    (c) 1996 Jan Kautz & Ulrich Gall
 *
 * Reads from a given file a list of files, which may
 * be loaded, if those files are pictures/sounds then
 * they may be displayed!
 */

package como.sys;

import como.io.*;
import como.awt.*;
import como.util.*;
import java.io.*;
import java.awt.*;
import java.util.*;
import java.applet.*;


public class RemoteFileDialog extends Panel {
	public static final int PICTURE = 0;
	public static final int SOUND = 1;
	public static final int DATA = 99;

	ComObj comobj;
	List list = null;
	Button okbutton;
	Button cancelbutton;
	Button playsound;
	ImageButton imagebutton;
	Label imagename;
	Component notify;
	Vector files = null;
	int selected = -1;
	int type;

	public RemoteFileDialog() {
		super();
	}

	/**
	 * The RemoteFileDialog is a helper-class, that opens a file on the server.
	 * (same pathes as the ComObj.openInputStream()). This files contains filenames
	 * that may be loaded. The RemoteFileDialog displays these filenames. You may
	 * select one of these. If it is a known filetype (picture/sound must be
	 * indicated in the first line of the filename-file), you can play/view
	 * with a double-click on the filename. The syntax of the file:
	 * first line: picture | sound | data.
	 * next lines: filename : description.
	 * @param comobj the ComObj.
	 * @param filename the name of the file that contains the filenames
	 * @param notify a Component that will be notified with an ACTION-Event,
	 *  where the target is set to the RemoteFileDialog and the arg to
	 *  the filename (null if cancel was pressed).
	 */
	public RemoteFileDialog( ComObj comobj, String filename, Component notify ) {
		super();

		this.notify = notify;
		this.comobj = comobj;

		setLayout( new BorderLayout( 10, 10 ) );
		list = new List( 8, false );

		DataInputStream dis = new DataInputStream( comobj.openInputStream( filename ) );
		files = StreamLine.loadLines( dis );

		if( files == null || files.size() < 1 )
		{
			type = DATA;
			add( "Center", new Label( "Error loading "+filename ) );
			add( "South", cancelbutton = new Button( "OK" ) );

			layout();
			repaint();

			return;
		}

		Enumeration e = files.elements();
		String tp = (String)e.nextElement();
		if( tp.equalsIgnoreCase( "picture" ) ) type = PICTURE;
		else if( tp.equalsIgnoreCase( "sound" ) ) type = SOUND;
		else type = DATA;

		while( e.hasMoreElements() ) {
			list.addItem( (String)e.nextElement() );
		}

		add( "Center", list );

		Panel buttonpanel = new Panel();
		buttonpanel.setLayout( new FlowLayout( FlowLayout.CENTER ) );
		buttonpanel.add( okbutton = new Button( "OK" ) );
		buttonpanel.add( cancelbutton = new Button( "Cancel" ) );
		add( "South", buttonpanel );

		switch( type ) {
			case SOUND:
				add( "North", new Label( "Double click for sound." ) );
				break;

			case PICTURE:
				add( "North", new Label( "Double click to preview picture." ) );

				Panel imagepanel = new Panel();
				imagepanel.setLayout( new BorderLayout( 10, 10 ) );
				imagepanel.add( "Center", imagebutton = new ImageButton( 70, 70, " pic" ) );
				imagebutton.setNoPush( true );
				imagepanel.add( "North", imagename = new Label( "(pic name)" ) );
				add( "East", imagepanel );
				break;

			default:
				break;
		}

		layout();
		repaint();
	}

	public boolean handleEvent( Event evt ) {
		if( evt.target == list ) {
			selected = list.getSelectedIndex();
		}

		return super.handleEvent( evt );
	}

	public String getName( String all ) {
		String name = "unknown";
		int idx;

		idx = all.indexOf( ':' );

		if( idx < 0 ) name = all;
		else name = all.substring( 0, idx-1 );

		return name.trim();
	}

	public boolean action( Event evt, Object what ) {
		if( evt.target == okbutton && selected != -1 ) {
			String name = getName( (String)files.elementAt( selected+1 ) );

			notify.postEvent( new Event( this, Event.ACTION_EVENT, name ) );
			return true;
		}

		if( evt.target == cancelbutton ) {
			notify.postEvent( new Event( this, Event.ACTION_EVENT, null ) );

			return true;
		}

		if( evt.target == list ) {
			String file = getName( (String)evt.arg );

			switch( type ) {
				case SOUND:
					AudioClip clip = comobj.loadAudioClip( file );
					clip.play();
					break;

				case PICTURE:
					// TODO: show name of the image somewhere
					Image img = comobj.loadImage( file );
					imagebutton.setImages( img, img );
					imagename.setText( file );
					break;
			}
		}

		return false;
	}
}
