/*
* @(#)ComObj.java    1.0 95/11/09 Ulrich Gall & Jan Kautz
*
* Copyright (c) 1996 Ulrich Gall & Jan Kautz 
* uhgall@cip.informatik.uni-erlangen.de
* Hofmannstr. 48, D-91052 Erlangen, Germany, Fax: +49-9131-201358
*
* Permission to use, copy, and distribute this software
* and its documentation for NON-COMMERCIAL purposes and without
* fee is hereby granted provided that this copyright notice
* appears in all copies. Please contact us for  further copyright 
* and licensing information.
*
* WE MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
* THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
* TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
* PARTICULAR PURPOSE, OR NON-INFRINGEMENT. WE SHALL NOT BE LIABLE FOR
* ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
* DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*/

package como.sys;

import java.util.*;
import java.awt.Image;
import java.applet.*;
import java.io.InputStream;

/**
* This is an interface for "Communicator" - objects that provides methods for Commlets 
* to send messages to the other Commlet instances.
*/
public interface ComObj {

	/**
	 * get the ID of myself.
	 * @return the local user's ID.
	 */
	public int getMyID();

	/**
	 * get the ID of master.
	 * @return the masters user's ID.
	 * -1 means: it is still unknown. Wait for the NEW_MASTER-message.
	 */
	public int getMasterID();

	/**
	* Get more information about the user with the given id.
	* @param id the id of the user.
	* @return the information about the User
	*/
	public User getUser( int id );	
	
	/**
	* get the name of given user with ID id.
	* @return the name of the user
	*/
	public String getUserName( int id );

	/**
	* get all Users in the commlet.
	* @return a Vector with all Users (User-Object)
	*/
	public Vector getUsers();

	/**
	* Sets the local User to u. Certain fields can not be changed, such as the nickname etc.
	* Those will simply stay the same
	* @param newUser. A User-Object containig the new information.
	*/
	public void setLocalUser(User newUser);

	/** 
	* Set a new topic. The Commlet has to decide if it may be set or not
	* -- if this method is called, the topic WILL be set.
	* @param topic the new topic.
	*/
	public void setNewTopic(String topic);

	/**
	* Returns true if this Commlet is the "Master". Exactly one Commlet instance in a
	* communication is "Master" at any given time.
	* @return true if master, false if not.
	*/
	public boolean iAmMaster();

	/**
	* Sends a message to msg.from
	* @param msg the msg you want to send.
	*/
	public void sendTo( Msg msg );

	/**
	* Sends a message to the other Commlet instances, but not to this one.
	* @param msg the msg you want to send.
	*/
	public void sendToOthers( Msg msg );

	/**
	* Sends a message to the Commlet instances contained in the byte array to[].
	* @param to array of user-ids where to send the message to.
	* @param msg the msg you want to send.
	*/
	public void sendToGroup(int[] to,Msg msg );
	
	/**
	* Sends a message to all Commlet instances, including this one.
	* @param msg the msg you want to send.
	*/
	public void sendToAll( Msg msg );

	/**
	* Sends msg to msg.to and blocks until msg.to answers.
	* The answer message is returned!
	* Do not use this method very often. It usually take's a long time
	* to return!!! If you respond to a ask-message, call msg.setAnswer( true );
	* @param msg the msg you want to send.
	* @return msg the answer-msg.
	*/
	public Msg ask( Msg msg );

	/**
	* Kick a User out of the channel. Do not use this too often :-)
	* Commlet-Writers: Only let the master kick a user, please.
	* @param id the id of the user
	* @param reason the reason why he was kicked out
	*/
	public void kickUser( int id, String reason );

	/**
	* Should be called by commlet.stop() to inform others that
	* the user is leaving to communication.
	*/
	public void logout();

	/**
	 * Load a picture with the specified filename!
	 * The ComObj has to know where to get it!
	 * @param filename the name of the file
	 * @return the Image.
	 */
	public Image loadImage( String filename );

	/**
	 * Load an audioclip with the specified filename!
	 * The ComObj has to know where to get it!
	 * @param filename the name of the file
	 * @return the AudioClip.
	 */
	public AudioClip loadAudioClip( String filename );

	/**
	 * Open an input-stream to the specified file!
	 * The ComObj has to know where to get it!
	 * @param filename the name of the file
	 * @return the InputStream for that file.
	 */
	public InputStream openInputStream( String filename );
}

