/*
* @(#)ServerIRC.java    1.0 96/03/03 Ulrich Gall & Jan Kautz
*
* Copyright (c) 1996 Ulrich Gall & Jan Kautz 
* uhgall@cip.informatik.uni-erlangen.de
* jnkautz@cip.informatik.uni-erlangen.de
* Hofmannstr. 48, D-91052 Erlangen, Germany, Fax: +49-9131-201358
*
*
* Server-IRC - it is used to do connections and receive invitations.
* This Server-Implementation uses the IRC-protocoll
*
*/

package como.irc;

import java.io.*;
import java.net.*;
import java.util.*;
import java.applet.Applet;
import como.sys.*;
import como.awt.*;
import como.io.*;
import como.commlet.*;
import como.commlet.draw.*;
import como.commlet.chat.*;
import como.commlet.nicnacnoe.*;
import como.commlet.superchat.*;
import como.commlet.scheduler.*;
import como.commlet.webcheckers.*;
import como.commlet.classroom.*;
import como.util.*;

public class ServerIRC extends Thread {
	static String PATH_COMO="como/";
	static String PATH_COMMLET=PATH_COMO+"commlets/";
	static String PATH_COMMLET_LIST = PATH_COMMLET + "commlets.txt";
	static String PATH_COMMLET_ICONS = PATH_COMMLET + "icons/";

	private IrcServerSocket waitSocket = null;
	private User ego;
	private Hashtable commlist;
	private ConstantsIRC constantsirc = new ConstantsIRC();

	CubbyHole UsersCubbyHole = new CubbyHole();
	CubbyHole ChannelsCubbyHole = new CubbyHole();

	ServerIRC( User e, String proxyhost, int port, ComoIRCFrontEnd cclient ) throws IOException {
		String localhost;

		try {
			localhost = (InetAddress.getLocalHost()).getHostName();
		} catch( Exception ex ) {
			Debug.msg( 20, "Server(): Couldn't get local host Name: "+ex.toString() );
			localhost = "localhost";
		}

		ego = e;
		commlist = new Hashtable();

		getConstantsIRC().setEgo( ego );
		getConstantsIRC().setLocalHost( localhost );
		getConstantsIRC().setProxyHost( proxyhost );
		getConstantsIRC().setProxyIRCPort( port );
		getConstantsIRC().setComoClient( cclient );		

		waitSocket = new IrcServerSocket( this );
		// exception must be caught from outside!

		// let's start our Thread
		start();
	}

	ConstantsIRC getConstantsIRC() {
		return constantsirc;
	}

	public void run() {
		Msg msg;
		boolean stopped = false;

		while( stopped == false )
		{
			msg = waitSocket.readMsg();

			if( msg.type == Msg.PING || msg.type == Msg.NO_DATA )
				continue;

			stopped = handleMsg( msg );
		}
	}

	public boolean handleMsg( Msg msg )
	{
		switch( msg.type )
		{
			case Msg.INVITATION:
				Hashtable inv = (Hashtable)msg.arg;
				IrcChan ch = (IrcChan)inv.get( "channel" );
				User from = (User)inv.get( "from" );
				String name = (String)from.get( User.NAME );

				getConstantsIRC().getComoClient().log( "Invitation for "+ch.commletname+" '"+ch.topic+"' from "+name );

				// ask user if user wants to accept invitation;
				getConstantsIRC().getComoClient().handleInvitation( from, ch );
				break;

			case Msg.ADD_USER:
				// someone joined #como-main
				Thread add = new UpdateThread( this, Msg.ADD_USER, (String)((User)msg.arg).get( User.NICK ) );
				add.start();
				break;

			case Msg.USER_LEFT:
				// someone left #como-main
				Thread left = new UpdateThread( this, Msg.USER_LEFT, (String)msg.arg );
				left.start();
				break;

			case Msg.LINE_DROPPED:
				new SmartFrame( "Sorry, line to Server dropped." );
				stopComObjs();
				try {
					waitSocket.close();
				} catch( Exception e ) { }

				// now show the login.
				getConstantsIRC().getComoClient().noConnection();

				return true;
			
			default:
				Debug.msg( 43, "ServerIRC.handleMsg(): Unknown Message-Type: " + msg.type );
				break;
		}

		return false;
	}

	/**
	 * startCommunication() starts a new commlet and invites the users[]
	 * 
	 */
	void startCommunication(String commletname, User users[], String topic ) {
		IrcChan chan = new IrcChan( commletname, topic );

		ConnectThreadIRC c = new ConnectThreadIRC( chan, this, ego, commlist, users );
		c.start();

		return;
	}

	/**
	 * This here is a bad thing. We wanted to avoid it, but we
	 * can't instantiate a ClassLoader! Therefore we have to
	 * instantiate the Commlets by hand!
	 */
	static Commlet loadCommlet( String commletname ) {
		if( commletname.compareTo("Draw") == 0 )
			return new Draw();

		if( commletname.compareTo("NicNacNoe") == 0 )
			return new NicNacNoe();
		
		if( commletname.compareTo("Chat") == 0 )
			return new Chat();

		if( commletname.compareTo("SuperChat") == 0 )
			return new SuperChat();

		if( commletname.compareTo("Scheduler") == 0 )
			return new Scheduler();

		if( commletname.compareTo("ClassRoom") == 0 )
			return new ClassRoom();
//  ajout de ces deux lignes
		if( commletname.compareTo("WebCheckers") == 0 )
			return new WebCheckers();
// fin des deux lignes
		Debug.msg( 98, "ServerIRC.loadCommlet(): "+commletname+" not found!" );

		return null;
	}

	/**
	 * Send an invitation for a specific channel to the user 'who'.
	 */
	public void sendInvitation( IrcChan chan, User who )
	{
		Hashtable inv = new Hashtable();

		inv.put( "channel", chan );
		inv.put( "from", getConstantsIRC().ego );

		waitSocket.writePrivMsg( who, new Msg( Msg.INVITATION, (Object)inv ) );
	}

	/** 
	 * Join the Channel 'chan'. Done asynchronously with a new thread.
	 */
	void joinChannel( IrcChan chan ) {
		ConnectThreadIRC c = new ConnectThreadIRC( chan, this, ego, commlist );
		c.start();
	}

	/* Called from ComoIRCFrontEnd. It stopps all ComObjs and
	 * closes the Socket to the ircd.
	 * This is called if the user presses "log out from como"
	 */
	public void logout() {
		// let's stop the main thread

		try {
			this.stop();
			waitSocket.close();
		} catch( Exception e ) {
			// noone cares here anymore
		}

		// now stop all commlets/comobjs
		stopComObjs();
	}

	/**
	 * Stops all ComObjs and their commlets.
	 */
	private void stopComObjs() {
		Enumeration e = commlist.keys();

		while( e.hasMoreElements() ) {
			ComObjIRC c = (ComObjIRC)e.nextElement();

			c.destroy();
			// will be removed from the list by commlet.destroy();
			// which calls server.loggedout( comobj, channel )
		}
	}

	/**
	 * Called by a ComObjIRC that quitted (user closed it).
	 * It just tells this the ServerIRC (me).
	 */
	public void loggedout( ComObjIRC c, IrcChan chan ) {
		if( commlist.get( c ) != null )
		{
			commlist.remove( c );
			getConstantsIRC().getComoClient().leftChannel( chan );
		}
		else
			Debug.msg( 33, "ServerIRC().loggedout(): Commlet #"+chan.ircName()+" already stopped." );
	}

	/**
	 * Return all users using our Programm, i.e. all that are connected
	 * to the IRCServer and who are in "pattern"
	 */
	synchronized User[] getUsers( String pattern, boolean como_only ) {
		Vector users;

		waitSocket.send( "WHO "+pattern, true );
		users = (Vector)UsersCubbyHole.get();

		User u[] = new User[users.size()];
		Enumeration e = users.elements();
		int i = 0;
		
		while( e.hasMoreElements() ) {
			u[i] = (User)e.nextElement();

			if( ((String)u[i].get( User.NAME )).startsWith( IrcSocket.NICK_PREFIX ) )
				u[i].put( User.COMOUSER, new Boolean(true) );

			i++;
		}
		
		return u;
	}

	/**
	 * Return all users on the specified channel
	 */
	User[] getUsersOnChannel( IrcChan ch ) {
		return getUsers( "#"+ch.ircName(), false );
	}

	/**
	 * Return all the channels (matching pattern)
	 */
	synchronized IrcChan[] getChannels( String pattern, boolean como_only ) {
		Vector channels;

		waitSocket.send( "LIST ", true );
		channels = (Vector)ChannelsCubbyHole.get();
		
		IrcChan c[] = new IrcChan[channels.size()];
		Enumeration e = channels.elements();
		int i = 0;
		
		while( e.hasMoreElements() ) {
			c[i] = (IrcChan)e.nextElement();
			i++;
		}
		
		return c;
	}

	/**
	 * Let's see if a user is on a channel.
	 */
	synchronized public boolean isUserOnChannel( User user, IrcChan channel ) {
		Enumeration e = commlist.elements();

		while( e.hasMoreElements() ) {
			IrcChan c = (IrcChan)e.nextElement();

			if( channel.ircName().equals( c.ircName() ) )
				return true;
		}

		return false;
	}

	/**
	 * Return the Applet, that started all. This is necessary,
	 * because the ComObj wants to load pictures and sound.
	 */
	Applet getApplet() {
		return getConstantsIRC().getComoClient().getApplet();
	}
}

/**
 * Update the display of ComoIRCClient.
 */
class UpdateThread extends Thread {
	int type;
	String what;
	ServerIRC server;

	public UpdateThread( ServerIRC s, int t, String w ) {
		type = t;
		what = w;
		server = s;
	}

	public void run() {
		switch( type )
		{
			case Msg.ADD_USER:
				server.getConstantsIRC().getComoClient().addUser( what );
				break;
			case Msg.USER_LEFT:
				server.getConstantsIRC().getComoClient().userLeft( what );
				break;
		}
	}
}


/**
 * Connect myself to an existing or a new channel.
 */
class ConnectThreadIRC extends Thread {
	private IrcChan chan;
	private User ego;
	private Hashtable commlist;
	private ServerIRC server;
	private User users[];
	private boolean connecttoexisting = true;

	ConnectThreadIRC( IrcChan c, ServerIRC s, User e, Hashtable cl ) {
		chan = c;
		ego = e;
		commlist = cl;
		server = s;

		connecttoexisting = true;
	}

	ConnectThreadIRC( IrcChan c, ServerIRC s, User e, Hashtable cl, User u[] ) {
		chan = c;
		ego = e;
		commlist = cl;
		server = s;
		users = u;

		connecttoexisting = false;
	}

	public void run()
	{
		IrcSocket socket;
		Commlet newcommlet;
		ComObjIRC com;
		User[] list = null;
		Msg msg;

		try {
			com = new ComObjIRC( server, chan, (User)server.getConstantsIRC().ego.clone(), !connecttoexisting );
		} catch( IOException e ) {
			// oops no connection to Server

			new SmartFrame( "Couldn't connect to server, try again later!" );

			server.getConstantsIRC().getComoClient().log( "Couldn't start Communication!" );
			return;
		}

		if( connecttoexisting )
			list = server.getUsersOnChannel( chan );

		server.getConstantsIRC().getComoClient().log( "Starting Commlet "+chan.commletname );

		/* AAAARGH: SECURITYMANAGER
		 *
		 * Is it really necessary to forbid an applet to create
		 * a ClassLoader. Netscape doesn't allow to read/write files
		 * to make sockets etc. No Class can do anything harmful.
		 * Then why can't I load a class over the net.
		 * I mean nothing else does netscape!!!! Why can't I???
		 * The whole advantage of JAVA is given away. Everybody
		 * says you just have to load those parts, that are needed.
		 * NICE IDEA. But this does not work. Neither with the
		 * appletviewer nor with Netscape!!!!
		 *
		 * Well then let's load some more Kb over the net just for fun!
		 */
		/*
		try {
			IRCClassLoader icl = new IRCClassLoader();
		   newcommlet = (Commlet)(new IRCClassLoader()).loadClass( chan.commletname, true ).newInstance();
		}
		catch( Exception e )
		{
			Debug.msg( 50, "ServerIRC.ConnectThread: Could not load Class over Net" );
			return;
		}
		*/
	

		// First load the Commlet
		// then set the corresponding com-Obj in the commlet
		// Then init() it. This sould work fine now. in
		// init() the ego-user is already known. The ComObj
		// does not deliver any messages before com.setCommlet()
		// is done. -> During the init() now other method from
		// the commlet will be called!

		newcommlet = ServerIRC.loadCommlet( chan.commletname );
		newcommlet.setCom( com );
		com.setCommletName( chan.commletname );
		newcommlet.init();
		com.setCommlet( newcommlet );
		newcommlet.showCommlet();

		commlist.put( com, chan );
		com.addMe( (User)server.getConstantsIRC().ego.clone() );

		if( connecttoexisting == true )
		{
			if( list.length == 1 ) {
				Msg newmastermsg = new Msg( Msg.NEW_MASTER, com.getMyID(), com.getMyID(),
											new Integer(com.getMyID()) );

				// tell comobj and commlet, that he is master then!
				com.preHandleMsg( newmastermsg );
				newcommlet.handleMsg( newmastermsg );
			}
			else
			{
				Vector users = com.getUsers();
				boolean adduser;

				// now add everyone who is already in this channel
				// and leave those who are already in the ComObj (via ADD_USER-Msg)
				for( int i = 0; i < list.length; i++ )
				{
					int id;

					adduser = true;
				
					Enumeration e = users.elements();
					while( e.hasMoreElements() )
					{
						User test = (User)e.nextElement();
						if( test.get(User.NICK).equals(list[i].get(User.NICK)) )
						{
							adduser = false;
							break;
						}
					}

					if( adduser == false )
						continue;

					id = com.loginUser( list[i] );
					if( id > 0 )				// only if user is allowed
						com.addUser( id );	// also tells commlet
				}
			}
		}
		else
		{
			// i am the master -> i may set the topic -> i will set the topic
			newcommlet.handleMsg( new Msg( Msg.NEW_TOPIC, com.getMyID(), com.getMyID(), chan.topic ) );
     
	  		// invite the users[]
			for( int i = 0; i < users.length; i++ )
				server.sendInvitation( chan, users[i] );
		}
	}
}


/*-----------------------------------------------------------------------
	HELP Classes used in Server.java
-----------------------------------------------------------------------*/

/**
 * This class does the main job here. It opens a connection back to the
 * irc-server then joins a channel. Then it reads message from that channel
 * and may write messages to that channel.
 */
class IrcSocket {
	final static int RPL_WHOREPLY   = 352;
	final static int RPL_ENDOFWHO   = 315;
	final static int RPL_NAMREPLY   = 353;
	final static int RPL_ENDOFNAMES = 366;
	final static int RPL_LIST       = 322;
	final static int RPL_LISTEND    = 323;

	final static int RPL_MOTDSTART = 375;
	final static int RPL_ENDOFMOTD = 376;

	final static int ERR_ERRONEUSNICKNAME = 432;
	final static int ERR_NICKNAMEINUSE = 433;
	final static int ERR_NICKCOLLISION = 436;

	final static int NO_MESSAGE = -1;
	final static int CMD_JOIN = 10001;
	final static int CMD_PART = 10002;
	final static int CMD_QUIT = 10003;
	final static int CMD_PRIVMSG = 10004;

	final static String NICK_PREFIX = "c_";

	String ret_user, ret_cmd, ret_msg, ret_channel, ret_address;
	protected Socket socket;
	DataInputStream input;
	DataOutputStream output;
	IrcChan chan;
	String nick;		// Someone must know the local nick name!

	ServerIRC server = null;

	/**
	 * Constructor for IrcSocket.
	 * Go to the main Channel.
	 */
	IrcSocket( ServerIRC s ) throws IOException {
		boolean ok;

		server = s;

		ok = Login( server.getConstantsIRC().proxyhost, server.getConstantsIRC().proxyircport,
				 	chan = new IrcChan( "como-main", "none", "Como Main Channel" ) );

		if( !ok ) throw new IOException();
		else {
			// This is the NICK he finally got!
			server.getConstantsIRC().ego.put( User.NICK, nick );
		}
	}
				 		
	/**
	 * Constructor for IrcSocket. Go to the the channel 'c'.
	 */
	IrcSocket( ServerIRC s, IrcChan c ) throws IOException {
		boolean ok;

		server = s;
		chan = c;

		ok = Login( server.getConstantsIRC().proxyhost, server.getConstantsIRC().proxyircport, c );

		if( !ok ) throw new IOException();
		else
			server.getConstantsIRC().ego.put( User.NICK, nick );
	}


	/**
	 * Connect myself to the IRC-Server and join a channel.
	 */
	private boolean Login( String host, int port, IrcChan chan ) {
		String name = (String)server.getConstantsIRC().ego.get( User.NAME );
		Socket s;
		int number = 1;

		number = ((int)(Math.random()*9000));
		nick = NICK_PREFIX + number;

		try {
			socket = new Socket( host, port );
			input  = new DataInputStream( socket.getInputStream() );
			output = new DataOutputStream( socket.getOutputStream() );
		} catch( Exception e ) {
			Debug.msg( 100, "IrcSocket(): " + e.toString() );
			return false;
		}

		send( "USER "+nick+" "+host+" "+host+" :"+name, true );
		send( "NICK "+nick, true );

		server.getConstantsIRC().getComoClient().log( "Waiting for Connection to Server." );

		while( true )
		{
			String line;
			int command;

			try {
				line = input.readLine();

				// oh line dropped
				if( line == null ) return false;
			} catch( Exception e ) {
				Debug.msg( 100, "IrcSocket.Login(): " + e.toString() );
				return false;
			}

			command = parseIrcMessage( line );

			if( command == ERR_ERRONEUSNICKNAME || command == ERR_NICKNAMEINUSE ||
				 	command == ERR_NICKCOLLISION )
			{
				// oh did not work let's try another one
				number++;
				nick = NICK_PREFIX + number;
				send( "NICK "+nick, true );
			}
			else if( command == RPL_ENDOFMOTD ) break;
		}

		// All User visible please!!!
		send( "MODE "+nick+" -i", true );
		send( "JOIN #"+chan.ircName(), true );

		while( true )
		{
			String line;
			try {
				line = input.readLine();

				// this here seems to be a "LINE_DROPPED"
				// message. *NO* IOException occures if the
				// connection drops!!!!
				if( line == null ) {
					return false;
				}
			} catch( Exception e ) {
				return false;
			}
			int command = parseIrcMessage( line );

			if( command == RPL_ENDOFNAMES ) // OK, I joined the channel
				break;
		}

		server.getConstantsIRC().getComoClient().log( "Connected." );
		return true;
	}


	/**
	 * Read a message from the channel belonging to that IrcSocket.
	 * Handles also those IRC-specific messages and tells the server
	 * or comobj, if they are of interest (like user added/left...).
	 */
	Msg readMsg() {
		int from, to, type, len;
		Object o;
		boolean stoploop = false;

		if( input == null ) return new Msg( Msg.NO_DATA );

		try {
			Vector users = new Vector();
			Vector channels = new Vector();

			while( !stoploop )
			{
			  	String line = input.readLine();
				int command;

				// ahh this means: Line Dropped!
				// well then throw an Exception, that i will
				// catch myself (see catch....)
				if( line == null ) throw new IOException();

			  	command = parseIrcMessage( line );

				// read as long as I get a message that is usefull for me.
			  	if( command == NO_MESSAGE ) continue;

				switch( command ) 
				{
			  	case CMD_PART:
			  		return new Msg( Msg.USER_LEFT, ret_user );

			  	case CMD_QUIT:
					return new Msg( Msg.USER_LEFT, ret_user );

			  	case CMD_JOIN:
					User user = new User();
					user.put( User.NICK, ret_user );

					// TODO:
					// well it's a little bit ugly perhaps.
					// but it works quite well

					user.put( User.NAME, "User nicknamed "+ret_user );
					return new Msg( Msg.ADD_USER, user );

				case RPL_WHOREPLY:
					{
						User u = new User();
						ret_msg = ret_msg.substring( 2, ret_msg.length() );
						u.put( User.NICK, ret_user );  // is nick
						u.put( User.NAME, ret_msg );
						u.put( User.ADDRESS, ret_address );
						users.addElement( u );
					}
					break;

				case RPL_ENDOFWHO:
					server.UsersCubbyHole.put( users );
					users = new Vector();
					break;

				case RPL_LIST:
					IrcChan ch = null;
					
					// TODO: Perhaps permit non-Como-Channels here
					// If yes: then uncomment this!
					// Otherwise: Every channel is a como-channel
					// if( ret_channel.startsWith( "como" ) ) { // ok is a como-channel

					if( true ) { // ok is a como-channel
						if( ret_channel.charAt(4) == '-' )	// ok this is como-main
						{
							// Do not show the como-main-channel where everybody
							// is in there! It is only for internal use!
							// ch = new IrcChan( ret_channel, "none", " " );
						}
						else
						{
							ch = new IrcChan();
							ch.fromString( ret_channel );
							ch.topic = ret_msg;
						}
					}
					else {
						ch = new IrcChan( ret_channel, "IRC", " " );
					}
					
					if( ch != null ) channels.addElement( ch );
					break;

				case RPL_LISTEND:
					server.ChannelsCubbyHole.put( channels );
					channels = new Vector();
					break;

				case CMD_PRIVMSG:
					if( ret_msg.startsWith( "COMO_" ) )
					{
						ret_msg = ret_msg.substring( 5, ret_msg.length() );
						stoploop = true;
					}
					else
					{
						// TODO: Use as chat message!!!
						// This is only necessary if non-Como-Channels
						// are allowed!
						Debug.msg( 30, "IrcSocket.readMsg(): got a irc-chat-message: "+ret_msg );
					}
					break;
				}
			}
		} catch( IOException e ) {
			// server.getConstantsIRC().getComoClient().log( "Connection to Server dropped." );
			Debug.msg( 1, "Exception: "+e );
			return new Msg( Msg.LINE_DROPPED, e.toString() );
		}

		StringTokenizer st = new StringTokenizer( ret_msg, "!" );
		from = (new Integer(st.nextToken())).intValue();
		to = (new Integer(st.nextToken())).intValue();
		type = (new Integer(st.nextToken())).intValue();

		if( st.nextToken().compareTo( "Y" ) == 0 )
		{
			ObjectInputStream ois = null;
			StringBufferInputStream stringstream = null;
			String s;
			Object object;

			s = st.nextToken( "\n" ); // Till the end of line!
			s = s.substring( 1, s.length() );

			stringstream = new StringBufferInputStream( s );
			ois = new ObjectInputStream( new IRCInputStream( stringstream ) );

			try {
				object = ois.readObject();
			}
			catch( Exception e ) {
				Debug.msg( 50, "IrcSocket.readMsg(): could not convert Object! Printing Trace: "+e.toString() );
				e.printStackTrace(System.out);
				return new Msg( Msg.NO_DATA );
			}

			if( object == null )
				Debug.msg( 70, "IrcSocket.readMsg(): read null-Object as arg!" );	

			return new Msg( type, from, to, object );
		}
		else
			return new Msg( type, from, to, null );
	}


	/**
	 * Write a message to all on that channel.
	 */
	boolean writeMsg( Msg msg ) {
		return writeMsg( msg, "#"+chan.ircName() );
	}


	/**
	 * Write a message to one user on that channel.
	 */
	boolean writePrivMsg( User to, Msg msg ) {
		return writeMsg( msg, (String)to.get( User.NICK ) );
	}


	/**
	 * Write a message to a user/all on that channel.
	 */
	synchronized private boolean writeMsg( Msg msg, String to ) {
		if( output == null ) return false;
		
		try {
			StringBuffer message = new StringBuffer( "PRIVMSG "+to+" :COMO_" );

			message.append( msg.from );
			message.append( "!" );
			message.append( msg.to );
			message.append( "!" );
			message.append( msg.type );
			message.append( "!" );
			if( msg.arg != null )
			{
				ObjectOutputStream oos = new ObjectOutputStream( new IRCOutputStream( output ) );

				message.append( "Y!" );
				output.writeBytes( message.toString() );
				oos.writeObject( msg.arg );
			}
			else {
				message.append( "N!" );
				output.writeBytes( message.toString() );
			}
			output.writeBytes( "\n" );

			return true;
		} catch( Exception e ) {
			Debug.msg( 98, "writeMsg(): " + e.toString() );
			return false;
		}
	}

	/**
	 * Parse a irc-message, but only parse those things that interest!
	 */
	synchronized int parseIrcMessage( String line ) {
		int colon, space, ex, pos, pos2;
		int commandnumber = NO_MESSAGE;
		int len;
		char ch;
		
		if( line.length() == 0 ) 
		{
			return NO_MESSAGE;
		}

		ch = line.charAt( 0 );
		if( ch != ':' ) {
			if( line.startsWith( "PING" ) ) {
				String out = "PONG "+server.getConstantsIRC().localhost;
				send( out.toString(), true );
			}
			return NO_MESSAGE;
		}
		
		// OK: Now it starts with ':'
		
		len = line.length();
		
		ex = line.indexOf( '!' );
		colon = line.indexOf( ':', 1 );
		space = line.indexOf( ' ' );
		
		if( space < 0 ) return NO_MESSAGE;
		
		pos = space+1;
		ch = line.charAt( pos );
		if( ch >= '0' && ch <= '9' ) {
			// Numeric Message
			
			pos2 = line.indexOf( ' ', pos );
			commandnumber = Integer.parseInt(line.substring(pos,pos2));
			
			pos = line.indexOf( ' ', pos2+1 );
			ret_user = line.substring( pos2+1, pos );

			if( commandnumber == RPL_WHOREPLY ) {
				StringTokenizer st = new StringTokenizer( line, " " );
				int i;
				for( i = 0; i < 8; i++ )
				{
					String str = st.nextToken();
					if( i == 5 ) ret_address = str;
					if( i == 7 ) ret_user = str;
				}
			}
			if( commandnumber == RPL_LIST ) {
				pos2 = line.indexOf( ' ', pos+1 );
				ret_channel = line.substring( pos+2, pos2 );
			}
		}
		else {
			// Has to be a message without numbers!!
			
			if( ex > 0 )
				ret_user = line.substring( 1, ex );
			else
				ret_user = line.substring( 1, pos );

			pos2 = line.indexOf(' ',pos);
			if( pos2 < 0 )
				return NO_MESSAGE;
				
			ret_cmd = line.substring( pos, pos2 );

			if( ret_cmd.equalsIgnoreCase( "JOIN" ) )
				commandnumber = CMD_JOIN;
			else if( ret_cmd.equalsIgnoreCase( "PART" ) )
				commandnumber = CMD_PART;
			else if( ret_cmd.equalsIgnoreCase( "QUIT" ) )
				commandnumber = CMD_QUIT;
			else if( ret_cmd.equalsIgnoreCase( "PRIVMSG" ) )
				commandnumber = CMD_PRIVMSG;
		}

		ret_msg = line.substring( colon+1, len );

		return commandnumber;
	}

	/**
	 * Send a String to this channel/IRC-Server.
	 * Used to send IRC-Commands.
	 */
	synchronized void send( String out ) {
		if( output == null ) return;

		try {
			output.writeBytes( out );
		} catch( Exception e ) {
			Debug.msg( 98, "IrcSocket.send(): Exception: "+e.toString() );
		}
	}
	
	/**
	 * Send a String to this channel/IRC-Server.
	 * Used to send IRC-Commands.
	 */
	void send( String out, boolean cr ) {
		if( cr == true )
			send( out+"\n" );
		else
			send( out );
	}
	
	/**
	 * Close the connection.
	 */
	void close() {

		try {
			output.flush();
		} catch( IOException e ) {
			Debug.msg( 98, "IrcSocket.close() output.flush(): "+e.toString() );
		}

		try {
			// wait a little bit!
			// Sometimes the last message wasn't sent, when
			// the socket was closed !!!

			Thread.sleep( 150 );
		} catch( Exception e ) {}
	
		try {
			socket.close();
		} catch( Exception e ) {
			Debug.msg( 98, "IrcSocket.close() socket: "+e.toString() );
		}

		try {
			input.close();
		} catch( Exception e ) {
			Debug.msg( 98, "IrcSocket.close() input: "+e.toString() );
		}

		try {
			output.close();
		} catch( Exception e ) {
			Debug.msg( 98, "IrcSocket.close() output: "+e.toString() );
		}

		input = null;
		output = null;
		socket = null;
	}
}

/**
 * Do as if this would be a ServerSocket :-)
 */
class IrcServerSocket extends IrcSocket {
	IrcServerSocket( ServerIRC server ) throws IOException {
		super( server );
	}
}

/**
 * I use many Constants in this file. I don't want/can hardcode
 * everything.
 */
class ConstantsIRC {
	User ego = new User();
	String proxyhost = "proxyhost";
	String localhost = "localhost";
	int proxyircport = 6667;
	ComoIRCFrontEnd comoclient = null;

	void setEgo( User e ) {
		ego = e;
	}
	
	void setProxyHost( String name ) {
		proxyhost = name;
	}
	
	void setLocalHost( String name ) {
		localhost = name;
	}
	
	void setProxyIRCPort( int p ) {
		proxyircport = p;
	}
	
	void setComoClient( ComoIRCFrontEnd c ) {
		comoclient = c;
	}

	ComoIRCFrontEnd getComoClient() {
		return comoclient;
	}
}

/* Does not work because of the SecurityManager
class IRCClassLoader extends ClassLoader {
	Hashtable cache = new Hashtable();

	IRCClassLoader() { }

	private byte loadClassData(String name)[] {
		byte b[];
		
		b = ProxyDataLoader.load( ConstantsIRC.proxyhost, ServerIRC.PATH_COMMLET, name );

		return b;
	}
	
	public synchronized Class loadClass( String name, boolean b ) {
		Class c = (Class)cache.get(name);

		if (c != null) return c;
		try {
			if ((c = findSystemClass(name)) != null) return c;
		}
		catch (Exception e) {
			c = null;
		}

		byte data[] = loadClassData(name);
		if( data == null ) return null;

		cache.put(name, c = defineClass(data, 0, data.length));
		return c;
	}
}

class ProxyDataLoader {
	ProxyDataLoader() {
	}
	
	static byte[] load( String host, String pfad, String name ) {
		URL url;
		
		try {
			InputStream input;
			byte b[];
			int len;
			
			url = new URL( "http:", host, pfad + name );
			input = url.openStream();
			len = input.available();
			b = new byte[len];
			if( input.read(b) != len )
				Debug.msg(90,"ProxyDataLoader().load(): loaded less than "+len+" bytes!" );
				
			return b;
		}
		catch( Exception e ) {
			Debug.msg(90,"ProxyDataLoader().load("+name+"): " + e.toString() );
			return null;
		}
	}
}
*/

