package como.irc;

import java.io.InputStream;
import java.io.IOException;

/**
 * Converts not-liked IRC-Characters to characters that an IRC-Daemon
 * understands
 */
public class IRCInputStream extends InputStream {
    /**
     * The actual input stream.
     */
    protected InputStream in;

    /**
     * Creates an input stream filter, filtering out and replacing escapechars
     * @param in	the input stream
     */
    public IRCInputStream(InputStream in) {
	this.in = in;
   }

    /**
     * Reads a byte. Will block if no input is available.
     * @return 	the byte read, or -1 if the end of the
     *		stream is reached.
     * @exception IOException If an I/O error has occurred.
     */
    public int read() throws IOException {
	int r= in.read();
	if (r == '\\')  {
	r = in.read() ;
	if (r == 'r') r = '\r';
	if (r == 'n') r = '\n';
	if (r == '0') r = '\0';
	// if r = // just leave it.
	}
	return r;
    }

    /**
     * Reads into an array of bytes.
     * Blocks until some input is available.
     * @param b	the buffer into which the data is read
     * @return  the actual number of bytes read, -1 is
     * 		returned when the end of the stream is reached.
     * @exception IOException If an I/O error has occurred.
     */
    public int read(byte b[]) throws IOException {
	return read(b, 0, b.length);
    }

    /**
     * Reads into an array of bytes.
     * Blocks until some input is available.
     * This method should be overridden in a subclass for
     * efficiency (the default implementation reads 1 byte
     * at a time).
     * @param b	the buffer into which the data is read
     * @param off the start offset of the data
     * @param len the maximum number of bytes read
     * @return  the actual number of bytes read, -1 is
     * 		returned when the end of the stream is reached.
     * @exception IOException If an I/O error has occurred.
     */
    public int read(byte b[], int off, int len) throws IOException {
	int i=0;
	for (i=0;i<len;i++)
		b[off+i] = (byte)read();
	return i;
    }

    /**
     * Skips bytes of input. 
     * @param n 	bytes to be skipped
     * @return	actual number of bytes skipped
     * @exception IOException If an I/O error has occurred.
     */
    public long skip(long n) throws IOException {
	byte[] b = new byte[(int)n];
	return read(b,0,(int)n);
    }

    /**
     * Returns the number of bytes that can be read.
     * without blocking.
     * @return the number of available bytes
     */
    public int available() throws IOException {
	return in.available();
    }

    /**
     * Closes the input stream. Must be called
     * to release any resources associated with
     * the stream.
     * @exception IOException If an I/O error has occurred.
     */
    public void close() throws IOException {
	in.close();
    }

    /**
     * Marks the current position in the input stream.  A subsequent
     * call to reset() will reposition the stream at the last
     * marked position so that subsequent reads will re-read
     * the same bytes.  The stream promises to allow readlimit bytes
     * to be read before the mark position gets invalidated.
     * @param readlimit the maximum limit of bytes allowed tobe read before the
     * mark position becomes invalid.
     */
    public synchronized void mark(int readlimit) {
	in.mark(readlimit);
    }

    /**
     * Repositions the stream to the last marked position.  If the
     * stream has not been marked, or if the mark has been invalidated,
     * an IOException is thrown.  Stream marks are intended to be used in
     * situations where you need to read ahead a little to see what's in
     * the stream.  Often this is most easily done by invoking some
     * general parser.  If the stream is of the type handled by the
     * parse, it just chugs along happily.  If the stream is *not* of
     * that type, the parser should toss an exception when it fails,
     * which, if it happens within readlimit bytes, allows the outer
     * code to reset the stream and try another parser.
     */
    public synchronized void reset() throws IOException {
	in.reset();
    }

    /**
     * Returns true if this stream type supports mark/reset
     */
    public boolean markSupported() {
	return false; // in.markSupported();
    }
}
