package como.commlet.draw;

import java.awt.*;
import java.util.*;
import java.io.IOException;
import como.sys.*;
import como.util.*;
import como.awt.*;
import como.io.*;
import como.commlet.*;

/*

Draw.java Whiteboard Commlet

(c) Jan Kautz und Ulrich Gall

For every user, there is a Group instance that contains the objects that user
has drawn. The local user may only change the objects in his group.
The GraphicsObject class hierarchy objects are very autonomouse-- they even
do the event handling themselves. They get the events from the canvas and
have their own handleEvent method that defines the way these objects are created. 
For example, a line is started by mouseDown, ended by mouseDrag and will be
accepted (sent to the others and added to the gorup) by mouseUp.

This way, the applet can easily expanded to support moving  and resizing objects.

*/

public class Draw extends WindowCommlet implements DrawObserver {

	DrawCanvas draw;	// a draw canvas that knows nothing about networking, but informs <this>
	DrawControls controls;
	// whenever its objects are changed

	static final int MSG_DRAW_NEW_OBJECT 	= 1001; // arg = new GO
	static final int MSG_DRAW_DEL_OBJECT 	= 1002; // arg = key
	static final int MSG_DRAW_CLEAR 	= 1003; // arg = key

	MenuItem miNew;
	MenuItem miCut;

	public String getCommletName() {
		return("Draw");
	}

	public String getCommletInfo() {
		return("Draw Commlet, V 1.0, Jan Kautz & Ulrich Gall");
	}

	public void init() 	{
		super.init();
		draw = new DrawCanvas(this);
		add("Center",draw);
		controls = new DrawControls(draw,this); // needs to tell draw what happens
		add("West",controls);
		draw.setControls(controls);

	Menu men= new Menu("File");
	miNew = new MenuItem("New");
	if (com.iAmMaster()) miNew.enable();
		else miNew.disable();
	men.add(miNew);
	MenuItem miOpen = new MenuItem("Open");
	MenuItem miSave = new MenuItem("Save");
	MenuItem miSaveAs = new MenuItem("Save As");
	miOpen.disable();
	men.add(miOpen);
	miSave.disable();
	men.add(miSave);
	miSaveAs.disable();
	men.add(miSaveAs);
	MenuBar mb = new MenuBar();
	mb.add(men);
	Menu menedit = new Menu("Edit");
	miCut = new MenuItem("Delete Last Object");
	miCut.disable();
	menedit.add(miCut);
	mb.add(menedit);
	setMenuBar(mb);

		pack();
	}


	public void createGroupFor(int who) {
		User u = com.getUser(who);
		String name = u.get(u.NAME).toString();
		if (!u.containsKey(u.COLOR))
			u.put(u.COLOR,Color.white);
		Color color  = (Color)u.get(u.COLOR);
		GOGroup grp = new GOGroup(name,color);
		grp.setFilled(true);
		draw.addGroup(new Integer(who),grp);
	}

	public void addUser(int who) { 
		createGroupFor(who);
		if (com.getMyID() == who) 
		draw.setCurrentGroupKey( new Integer(com.getMyID()) );
		else {
			// Somebody other than myself joined us, so I will send him the contents of my GOGroup
			Enumeration e = ((GOGroup)draw.getGroup( new Integer(com.getMyID()) ) ).elements();
			while (e.hasMoreElements()) {
				Msg m = new Msg(MSG_DRAW_NEW_OBJECT,(GraphicsObject)e.nextElement());
				m.to = who;
				com.sendTo(m);				
			}
			if (draw.colormode == draw.COLORMODE_OWNER) {
				 controls.updateColormodepanel();
			}
		}
	}

	public void userLeft(int who) { 
	// TODO 
	draw.getGroup(new Integer(who)).setFilled(false);
	controls.updateColormodepanel();
	}
	public void newObject(GraphicsObject g) {
		com.sendToOthers(new Msg(MSG_DRAW_NEW_OBJECT,g));
		miCut.enable();
	}

	public void delObject(Object key) {
		com.sendToOthers(new Msg(MSG_DRAW_DEL_OBJECT,key));
		if ((new Integer(1)).equals(key)) miCut.disable();
	}

	public boolean handleMsg(Msg msg)	{
		if (super.handleMsg(msg)) return true;
		String name = com.getUserName(msg.from);
		if (msg.type == MSG_DRAW_NEW_OBJECT)
		{
			// If this is a GOImage, load it. Not relly pretty, but what else can I do?
			if (msg.arg instanceof GOImage) {
				((GOImage)msg.arg).setData(com,draw);
			}

			draw.appendToGroup(new Integer(msg.from),(GraphicsObject)msg.arg);
			//			draw.paintObject((GraphicsObject)msg.arg);
			draw.repaint();
			return true;
		}
		else if (msg.type == MSG_DRAW_CLEAR) {
			draw.removeGroups();	
			Enumeration e = com.getUsers().elements();
			while (e.hasMoreElements()) {
				int id = ((User)e.nextElement()).getID();
				createGroupFor(id);
			}
			controls.updateColormodepanel();
		}
		else if (msg.type == MSG_DRAW_DEL_OBJECT)
		{
			draw.delFromGroup(new Integer(msg.from),msg.arg); 
			draw.repaint();
			return true;
		}
		else if (msg.type == Msg.NEW_MASTER) {
			miNew.disable();
			if (((Integer)msg.arg).intValue() == com.getMyID())
				miNew.enable();
		}
		else if (msg.type == Msg.NEW_USER_INFO)
		{
			User u = (User)com.getUser(msg.from);
			draw.getGroup(u.get(u.ID)).name = u.get(u.NAME).toString();
			draw.getGroup(u.get(u.ID)).setColor((Color)u.get(u.COLOR));
			controls.updateColormodepanel();
			draw.repaint();
			return true;
		}
		return false;
	}
	public boolean action(Event evt, Object what) {
		if (evt.target == miNew) {
			com.sendToAll(new Msg(MSG_DRAW_CLEAR,null));
			return true;
		}
		if (evt.target == miCut) {
			draw.doCut();
			return true;
		}
	return false;
	}
	public boolean handleEvent(Event evt) {
	if (evt.id == Event.WINDOW_DESTROY) {
		draw.setCurrentGO(null);
	}
	return super.handleEvent(evt);
	}
	public Image loadImage(String n) {
		return com.loadImage(n);	
	}
	public ComObj getCom() {
		return com;
	}
}

/**********************
* DrawCanvas
*
*
* The following methods are called by the Commlet
*
* addGroup
* appendToGroup
* delFromGroup
* getGroup
* setCurrentGroupKey
* setControls
*
* observer is informed about changes in objects. 
* User actions may only alter currentgroup
*
*/

class DrawCanvas extends Canvas
{
	public static final int EVENT_ACCEPT = 150173;

	GOGroup 		objects;
	DrawObserver 	observer;
	DrawControls 	controls;

	GraphicsObject currentGO;
	Object 			currentGroupKey;

	String 			colormode;

	// Color Modes
	static String COLORMODE_OWNER = "Owner's";
	static String COLORMODE_OBJECT = "Object's";

	// keys
	static final int KEY_PASTE 		= 112;
	static final int KEY_CUT 			= 127;
	static final int KEY_COPY 		= 99;
	static final int KEY_DUMP 		= 68;

	public DrawCanvas(DrawObserver d)
	{
		setBackground(Color.black);
		colormode = COLORMODE_OBJECT;
		observer = d;
		objects = new GOGroup();
		currentGO = null;
		currentGroupKey = null;
	}

	public void setControls(DrawControls c) {
		controls = c;
	}

	public void setCurrentGroupKey(Object group)
	{
		currentGroupKey = group;
	}

	public GraphicsObject getCurrentGroup()
	{
		return (GraphicsObject)objects.get(currentGroupKey);
	}

	public void setCurrentGO(GraphicsObject go)
	{
		if (currentGO != null) currentGO.handleEvent(new Event(this,Event.WINDOW_DESTROY,null));
		currentGO = go;
	}

	public void setColormode(String m)
	{
		colormode = m;	
		repaint();
	}
	public Dimension preferredSize() {
		return new Dimension(640,480);
	}
	public void paint(Graphics g)
	{
		Enumeration e = objects.keys();
		while (e.hasMoreElements())
		{
			Object o = e.nextElement();
			GraphicsObject go = (GraphicsObject)objects.get(o);
			if (colormode.compareTo(COLORMODE_OBJECT) == 0) go.draw(g); 
			else go.draw(g,go.getColor());
		}
		if (currentGO != null) {
			if (colormode.compareTo(COLORMODE_OBJECT) == 0) currentGO.draw(g); 
			else currentGO.draw(g,getCurrentGroup().getColor());
		}
	}

	public void paintObject(GraphicsObject g)
	{
		if (getGraphics() != null) {
			if (g != null) {
				if (colormode.compareTo(COLORMODE_OBJECT) == 0) currentGO.draw(getGraphics()); 
				else currentGO.draw(getGraphics(),getCurrentGroup().getColor());
			}
		}
	}

	/*************************************************************
	* Adding and removing objects
	*/
	public void addGroup(Object key,GraphicsObject g)
	{
		objects.put(key,g);
		paintObject(g);
	}
	public void removeGroups() {
		objects = new GOGroup();
		repaint();
	}

	public GraphicsObject getGroup(Object key)
	{
		return (GraphicsObject)objects.get(key);
	}

	public void appendToGroup(Object key,GraphicsObject g) 
	{
		GraphicsObject o = objects.get(key);
		if (o instanceof GOGroup) {
			GOGroup group = (GOGroup)o;
			group.put(new Integer(group.size()+1),g);
			paintObject(g);
		}
		else {
			Debug.msg(90,"DrawPanel.appendToGroup: Not a group: "	+ o.toString());
		}
	}
	public void delFromGroup(Object gk,Object key)
	{
		GOGroup g = (GOGroup) objects.get(gk);
		g.remove(key);
		repaint();
	}
	public void acceptCurrentGO() {
						appendToGroup(currentGroupKey	,currentGO);
						observer.newObject(currentGO);
						paintObject(currentGO);
	}
	public void doCut() {
		GOGroup g = ((GOGroup)objects.get(currentGroupKey));
		if (g.size() > 0) {
			observer.delObject(new Integer(g.size()));
			delFromGroup(currentGroupKey, new Integer(g.size()));
		}
	}
	/******************************************************************************************
	* Event handling
	*/
	public boolean handleEvent(Event evt) {
		// We want keyboard events if the mouse enters...
		// under some platforms, we need focus for that.
		// But, under W95, requestFocus() also pops the window
		// up, so we don't want to do this under W95
		if ((!System.getProperty("os.name","").startsWith("Windows"))
			 && (evt.id == Event.MOUSE_ENTER)) {
			requestFocus();
			return true;
		}
		if ((evt.id == Event.KEY_PRESS) && (evt.key == 127)) {
			doCut();
			return true;
		}
		if ( 	(currentGO != null) && (
			 ((evt.x > 0) && (evt.y > 0)) ||
			 (evt.id == Event.KEY_PRESS)    )
		   )  {
			{
				if (currentGO.handleEvent(evt))
				{
					if (evt.id == EVENT_ACCEPT)
					{
						// This means we should accept the currentGO and start a new one
						acceptCurrentGO();
						currentGO = currentGO.getNew();
						currentGO.color = controls.currentcolor;
						currentGO.filled = controls.filled;
						//currentGO.moveTo(evt.x,evt.y);
					}
					paintObject(currentGO);
					return true;
				}
			}
		}
		return false;				
	}
}

/*******************************************************************************************
* DrawControls
*/

class DrawControls extends Panel {

	// modes
	static int MODE_POLYGON 	= 0;
	static int MODE_TEXT		= 1;
	static int MODE_LINE		= 2;
	static int MODE_RECT		= 3;
	static int MODE_OVAL		= 4;
	static int MODE_IMAGE		= 5;

	static int DRAWMODESIZE = 64;

	DrawCanvas drawcanvas;
	DrawObserver draw;

	RadioImageButton drawmodesel;
	Choice	colormodeChoice;

	Panel colormodepanel;
	Panel drawmodepanel;
		
	Checkbox filledCB;
	ColorSelector colorsel;
		
	boolean filled;
	Color currentcolor;
	Color fillColor;

	public DrawControls(DrawCanvas dc,DrawObserver dob) {
		drawcanvas = dc;
		draw = dob;
			
		setLayout(new VertLayout(VertLayout.STRETCH));

		drawmodesel = new RadioImageButton();
		drawmodesel.setLayout(new GridLayout(3,2));
		drawmodesel.add(draw.loadImage("imgPOLYGON"),draw.loadImage("imgPOLYGONP"),DRAWMODESIZE,DRAWMODESIZE);
		drawmodesel.add(draw.loadImage("imgTEXT"),draw.loadImage("imgTEXTP"),DRAWMODESIZE,DRAWMODESIZE);
		drawmodesel.add(draw.loadImage("imgLINE"),draw.loadImage("imgLINEP"),DRAWMODESIZE,DRAWMODESIZE);
		drawmodesel.add(draw.loadImage("imgRECT"),draw.loadImage("imgRECTP"),DRAWMODESIZE,DRAWMODESIZE);
		drawmodesel.add(draw.loadImage("imgOVAL"),draw.loadImage("imgOVALP"),DRAWMODESIZE,DRAWMODESIZE);
		drawmodesel.add(draw.loadImage("imgIMAGE"),draw.loadImage("imgIMAGEP"),DRAWMODESIZE,DRAWMODESIZE);
		drawmodesel.switchButtonOn(MODE_POLYGON);
		Panel p = new Panel();
		p.setLayout(new BorderLayout());
		p.add("West",drawmodesel);
		add(p);

		drawmodepanel = new Panel();
		add(drawmodepanel);

		currentcolor = Color.white;
		add(new Label("Current Color"));
		colorsel = new ColorSelector(currentcolor);
		add(colorsel);
			
		filled =	false;
		filledCB = new Checkbox("Filled");
		filledCB.setState(filled);
		add(filledCB);

		add(new Label("Color Mode"));
		colormodeChoice = new Choice();
		colormodeChoice.addItem(dc.COLORMODE_OBJECT);
		colormodeChoice.addItem(dc.COLORMODE_OWNER);
		add(colormodeChoice);
		colormodepanel = new Panel();
		updateColormodepanel();
		add(colormodepanel);
         
		GraphicsObject cur = new GOPolygon();
		cur.setColor(currentcolor);
		cur.setFilled(filled);
		setCurrentGO(cur);

	}

	public void setCurrentGO(GraphicsObject cur)
	{
		cur.layoutPropPanel(drawmodepanel);			
		layout();
		paintAll(getGraphics());
		drawcanvas.setCurrentGO(cur);
	}

	public void setCurrentColor(Color c) {
		currentcolor = c;
		if (drawcanvas.currentGO != null) drawcanvas.currentGO.setColor(c);
	}

	public void updateColormodepanel() {
		colormodepanel.removeAll();
		if (drawcanvas.colormode.compareTo(DrawCanvas.COLORMODE_OWNER) == 0)
		{
			colormodepanel.setLayout(new VertLayout(VertLayout.STRETCH));
			Enumeration e = drawcanvas.objects.elements();
			while (e.hasMoreElements()) {
				GraphicsObject g  = (GraphicsObject)e.nextElement();
				ColorView c = new ColorView(g.getColor(),g.name);
				if (g.filled) c.enable(); else c.disable();
				colormodepanel.add(c);
			}
		}
		layout();
		colormodepanel.paintAll(colormodepanel.getGraphics());
	}

	public boolean action(Event evt,Object arg) {
		
		if (evt.target == drawmodesel) {
			int md = ((Integer)evt.arg).intValue();
			GraphicsObject cur = null;
			if (drawcanvas.currentGO instanceof GOText) drawcanvas.acceptCurrentGO();
			if (md == MODE_POLYGON) cur = new GOPolygon();
			if (md == MODE_RECT) cur = new GORect();
			if (md == MODE_LINE) cur = new GOLine();
			if (md == MODE_TEXT) cur = new GOText();
			if (md == MODE_OVAL) cur = new GOOval();
			if (md == MODE_IMAGE) {
				GOImage img = new GOImage();
				// set name and load image
				ComObj c = draw.getCom();
				String n = "default";
				img.setData(c,drawcanvas);
				cur = img;
			}

			cur.setColor(currentcolor);
			cur.setFilled(filled);
			setCurrentGO(cur);
			return true;
		}
		if (evt.target == colormodeChoice)  {
			drawcanvas.setColormode(colormodeChoice.getSelectedItem());
			if (colormodeChoice.getSelectedItem().compareTo(DrawCanvas.COLORMODE_OWNER) == 0)
				currentcolor = drawcanvas.getCurrentGroup().getColor();
			else
				currentcolor = colorsel.getColor();
			updateColormodepanel();
			return true;
		}
		if (evt.target == filledCB) {
			filled = filledCB.getState();
			if (drawcanvas.currentGO != null) {
				drawcanvas.currentGO.setFilled(filled);
				drawcanvas.paintObject(drawcanvas.currentGO);
			}
		}
		if (evt.target == colorsel) {
			setCurrentColor(colorsel.getColor());
		}
		return false;
	}

	public boolean handleEvent(Event evt) {
		boolean sup =  super.handleEvent(evt);
		// TODO: This is not necessarily for currentGO
		GraphicsObject cur = drawcanvas.currentGO;
		boolean b = false;
		if (cur != null) {
			b = cur.handlePropPanelEvent(evt,drawcanvas);
		}
		return (sup || b);
	}
}

class ColorView extends Canvas {
	String name;
	static final int SPACE_BOTTOM = 6;
	static final int SPACE_LEFT = 10;
	boolean enabled = true;

	public ColorView(Color c, String n) {
		name = n;
		setBackground(c);
		if ((c.getRed()+c.getGreen()+c.getBlue()) < 384) setForeground(Color.white);
			else setForeground(Color.black);
	}

	public void paint(Graphics g) {
		Font f = getFont();
		if (enabled) {
		g.setFont(new Font(f.getName(),f.BOLD,f.getSize()));
		}
		else {
		g.setFont(new Font(f.getName(),f.ITALIC,f.getSize()));
		}
		g.drawString(name,SPACE_LEFT,size().height-SPACE_BOTTOM);
	}

	public Dimension preferredSize() {
		FontMetrics fm = getFontMetrics(getFont());	
		return new Dimension(2*SPACE_LEFT+fm.stringWidth(name),
				fm.getHeight()+2*SPACE_BOTTOM);
	}
	public void enable() {
	super.enable();
	enabled = true;
	}
	public void disable() {
	super.disable();
	enabled = false;
	}

	public Dimension minimumSize() {
		return preferredSize();
	}
}

/******************************************************************************************/
interface DrawObserver {

	void newObject(GraphicsObject g);
	void delObject(Object key);
	Image loadImage(String name);
	ComObj getCom();
}
