/*
* @(#)WindowCommlet.java    1.0 95/11/09 Ulrich Gall & Jan Kautz
*
* Copyright (c) 1996 Ulrich Gall & Jan Kautz 
* uhgall@cip.informatik.uni-erlangen.de
* Hofmannstr. 48, D-91052 Erlangen, Germany, Fax: +49-9131-201358
*
* Permission to use, copy, and distribute this software
* and its documentation for NON-COMMERCIAL purposes and without
* fee is hereby granted provided that this copyright notice
* appears in all copies. Please contact us for  further copyright 
* and licensing information.
*
* WE MAKE NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
* THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
* TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
* PARTICULAR PURPOSE, OR NON-INFRINGEMENT. WE SHALL NOT BE LIABLE FOR
* ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
* DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
*/
package como.commlet;

import java.awt.*;
import java.util.Date;
import como.sys.*;
import como.util.*;

/**
 * Any Commlets that might be used within other commlets should be derived
 * from WindowCommlet. This is a Frame that implements Commlet.
 * 
 * Also, it can be put into a MetaCommlet. 
 */
public class WindowCommlet extends Frame implements Commlet 
{
	protected ComObj com;
	private MetaCommlet metacommlet = null;

	/**
	* To make it possible to set a topic, simply place this TextField anywhere in this Frame.
	* If WindowCommlet.handleEvent receives an Action from it, it sets a new topic. If a new topiv
	* is set, WindowCommlet.handleMsg() automatically updates the TextField.
	*/
	private  TextField topicTF;

	public String getSessionName()
	{
		return(getCommletName() + " started on " + (new Date()).toString());
	}

	public String getCommletName()
	{
		return("Base class Windowcommlet");
	}

	public String getCommletInfo()
	{
		return("Part of the native commlets, V 1.0, Jan Kautz & Ulrich Gall");
	}

	/**
	 * Returns a TextField where the user may type in the topic. If an action
	 * occurs within it, the text is taken over as a new topic and sent to the other
	 * commlet sessions, which then upate the contents of their topic TextField.
	 */
	protected TextField getTopicTextField() {
		return topicTF;
		}

	/**
	 * This also takes care of a the menu bar of its MetaCommlet, if it exists.
	 * If it's not a metacommlet calls stop(). Else it only hides the frame.
	 */
	public void hideCommlet()
	{
		hide();
		if (metacommlet != null)
		{
			metacommlet.setStateOfMenuItemBelongingTo(this,false);
		}
		else stop();
	}

	/**
	 * Shows the commlet.
	 * This also takes care of a the menu bar of its MetaCommlet, if it exists.
	 */
	public void showCommlet()
	{
		show();
		if (metacommlet != null)
		{
			metacommlet.setStateOfMenuItemBelongingTo(this,true);
		}
	}

	public void setMetaCommlet(MetaCommlet m)
	{
		metacommlet = m;
	}

	public void setCom(ComObj c)
	{
		com = c;
	}

	public void init()
	{
		setTitle(getSessionName());	
		topicTF = new TextField("No topic set");
	}

	/**
	 * This destroys the local Commlet session. Automatically
	 * called. If you ovveride this, then you have to call
	 * super.stop() at the end of your method!
	 */
	public void stop()
	{
		if (metacommlet == null) {
			com.logout();
		}
		dispose();
	}

	/**
	 * Override this for new behaviour!
	 */
	public boolean isUserAdmitted(User u)
	{
		return true;
	}

	/**
	 * If a user leaves, this method will be called.
	 * @param who the id of the user.
	 */
	public void userLeft(int who) {}

	/**
	 * If a user comes, this method will be called.
	 * @param who the id of the user.
	 */
	public void addUser(int who) {}

	/**
	 * This is a default handleMsg() that calles the helper-methods
	 * userLeft() and addUser(). And it takes care of the
	 * NEW_TOPIC-message. At the moment you can be sure, that this
	 * method is not called more than once at a time. But in future
	 * this might change! If you need this behaviour then
	 * make it synchronized!
	 */
	public boolean handleMsg(Msg msg)
	{
		if (msg.type == Msg.ADD_USER) {
			addUser(((Integer)msg.arg).intValue());
			return true;
		}
		if (msg.type == Msg.USER_LEFT) {
			userLeft(((Integer)msg.arg).intValue());
			return true;
		}
		if (msg.type == Msg.NEW_TOPIC) {
			topicTF.setText(msg.arg.toString());
			return true;
		}
		return false;
	}

	/**
	 * If you close the window, it will call automatically hide() or
	 * dispose the commlet (depending if you are in a metacommlet or not).
	 * It also sets the topic (and send a NEW_TOPIC message), if the topic
	 * is changed in the topic-TextField.
	 */
	public boolean handleEvent(Event evt)
	{
		if (evt.id == Event.WINDOW_DESTROY)
		{
			hideCommlet();
			return true;
		}
		if ((evt.id == Event.ACTION_EVENT) && (evt.target == topicTF)) {
			com.setNewTopic(topicTF.getText());
		}
		return super.handleEvent(evt);
	}
}
