package como.awt;

/*
 * @(#)VertLayout.java	1.0 96/03/06 Jan Kautz
 *
 */

import java.awt.*;

/**
 * Vert layout is used to layout buttons in a panel. It will arrange
 * buttons top down.
 * Each line is aligned left/right/centered.
 *
 * @version 	1.0
 * @author 	Jan Kautz
 */
public class VertLayout implements LayoutManager {
    public static final int LEFT 	= 0;
    public static final int CENTER 	= 1;
    public static final int RIGHT 	= 2;
    public static final int STRETCH 	= 3;

    int align;
    int hgap;
    int vgap;

    /**
     * Constructs a new Vert Layout with a left alignment.
     */
    public VertLayout() {
	this(LEFT, 5, 5);
    }

    /**
     * Constructs a new Vert Layout with the specified alignment.
     * @param align the alignment value
     */
    public VertLayout(int align) {
	this(align, 5, 5);
    }

    /**
     * Constructs a new Vert Layout with the specified alignment and gap
     * values.
     * @param align the alignment value
     * @param hgap the horizontal gap variable
     * @param vgap the vertical gap variable
     */
    public VertLayout(int align, int hgap, int vgap) {
	this.align = align;
	this.hgap = hgap;
	this.vgap = vgap;
    }

    /**
     * Adds the specified component to the layout. Does not apply.
     * @param name the name of the component
     * @param comp the the component to be added
     */
    public void addLayoutComponent(String name, Component comp) {
    }

    /**
     * Removes the specified component from the layout. Does not apply.
     * @param comp the component to remove
     */
    public void removeLayoutComponent(Component comp) {
    }

    /**
     * Returns the preferred dimensions for this layout given the components
     * in the specified target container.
     * @param target the component which needs to be laid out
     * @see Container
     * @see #minimumSize
     */
    public Dimension preferredLayoutSize(Container target) {
	Dimension dim = new Dimension(0, 0);
	int nmembers = target.countComponents();

	for (int i = 0 ; i < nmembers ; i++) {
	    Component m = target.getComponent(i);
	    if( m.isVisible() ) {
		Dimension d = m.preferredSize();
		dim.height += d.height;
                dim.width = Math.max( dim.width, d.width );

		if (i > 0) {
		    dim.height += vgap;
		}
	    }
	}
	Insets insets = target.insets();
	dim.width += insets.left + insets.right + hgap*2;
	dim.height += insets.top + insets.bottom + vgap*2;
	return dim;
    }

    /**
     * Returns the minimum dimensions needed to layout the components
     * contained in the specified target container.
     * @param target the component which needs to be laid out 
     * @see #preferredSize
     */
    public Dimension minimumLayoutSize(Container target) {
	Dimension dim = new Dimension(0, 0);
	int nmembers = target.countComponents();

	for (int i = 0 ; i < nmembers ; i++) {
	    Component m = target.getComponent(i);
	    if( m.isVisible() ) {
		Dimension d = m.minimumSize();
		dim.height += d.height;
                dim.width = Math.max( dim.width, d.width );

		if (i > 0) {
		    dim.height += vgap;
		}
	    }
	}
	Insets insets = target.insets();
	dim.width += insets.left + insets.right + hgap*2;
	dim.height += insets.top + insets.bottom + vgap*2;
	return dim;
    }

    /**
     * Lays out the container. This method will actually reshape the
     * components in target in order to satisfy the constraints of
     * the VertLayout object. 
     * @param target the specified container being laid out.
     * @see Container
     */
    public void layoutContainer(Container target) {
	Insets insets = target.insets();
	Dimension targetsize = target.size();
	int maxwidth = targetsize.width - (insets.left + insets.right + hgap*2);
	int nmembers = target.countComponents();
	int x = hgap, y = insets.top + vgap;
	int rowh = 0, start = 0;

	for (int i = 0 ; i < nmembers ; i++) {
	    Component m = target.getComponent(i);
	    if( m.isVisible() ) {
		Dimension d = m.preferredSize();
		Dimension willbe = new Dimension( d );

		if( d.width > maxwidth )     // zu breit!
		    willbe.width = maxwidth;

		if( align == STRETCH )     // muss so breit sein
		    willbe.width = maxwidth;

		m.resize( willbe.width, willbe.height );

		switch( align )
		{
		   case CENTER:
		      m.move( hgap + (maxwidth - willbe.width)/2, y );
		      break;

		   case RIGHT:
		      m.move( hgap + maxwidth - willbe.width, y );
		      break;

		   case STRETCH:
		   case LEFT:
		      m.move( x, y );
		      break;
		}

		y += willbe.height + vgap;
	    }
	}
    }
    
    /**
     * Returns the String representation of this FlowLayout's values.
     */
    public String toString() {
	String str = "";
	switch (align) {
	  case LEFT:    str = ",align=left"; break;
	  case CENTER:  str = ",align=center"; break;
	  case RIGHT:   str = ",align=right"; break;
	  case STRETCH: str = ",align=stretched"; break;
	}
	return getClass().getName() + "[hgap=" + hgap + ",vgap=" + vgap + str + "]";
    }
}
