/*
 * @(#)ImageButton.java  (c) 1996 Jan Kautz
 */

package como.awt;

import como.util.*;
import java.awt.*;
import java.awt.image.*;

/**    
 * ImageButton is a button which uses images to represent
 * the states of the button - normal, pressed.
 * They don't have to be fully loaded! They will be loaded
 * asynchronously here!
 */

public class ImageButton extends Canvas {
    Image normal;
    Image pressed;
    Image current;
    boolean down = false;
    boolean focus = false;
    boolean isswitch = false;
    boolean nopush = false;
    int width, height;
    String text;

    /**
     * Construct an image button. We have two states:
     * Normal and pressed. They both must be non-null.
     * Then you must specify the width and height of the button.
     * (May be different from the icture's size and the picture
     * will be loaded asynchronously!)
     */
    public ImageButton( Image normal, Image pressed, int w, int h ) {
    	width = w;
	height = h;

	down = false;
	
	// set my current size to width/height
	resize( width, height );
	
	setImages( normal, pressed );
    }

    public ImageButton( int w, int h, String text ) {
    	width = w;
	height = h;
	down = false;

	this.normal = null;
	this.pressed = null;
	this.text = text;
	this.current = null;

	drawIt();
    }

    /**
     * Here you may change the pictures used in the button.
     */
    public void setImages( Image normal, Image pressed ) {
	this.normal = normal;
	this.pressed = pressed;

	this.current = normal;
	
	// let's load and scale the image
	if( normal != null )
	    prepareImage( normal, width, height, this );

	if( pressed != null )
	    prepareImage( pressed, width, height, this );

	drawIt();
    }

    /**
     * Make it a Switch instead of a Button. Now the user can press
     * the switch and it's on or off then! You will get an action-message
     * that the state changed. Get state with getState().
     */
    public void setSwitch( boolean sw ) {
    	isswitch = sw;
	drawIt();
    }

    /**
     * Here you can set, if you want the button to be pushed
     * down, if someone clicks or not. 
     * (Usefull if you just want to display a picture!)
     */
    public void setNoPush( boolean np ) {
    	nopush = np;
	drawIt();
    }

    /**
     * Set the current state of the switch!
     * (If it is a button then do nothing!)
     * @param onoff indicating on or off
     */
    public void setState( boolean onoff ) {
    	if( isswitch ) {
	    down = onoff;

	    if( onoff == true )
	    	current = pressed;
	    else
	    	current = normal;

	    drawIt();
	}
    }
 
    /**
     * Get the current state of the switch! (if it's a button
     * you will get true if it's currently pressed).
     * @return if the switch is on then return true; else false.
     */
    public boolean getState() {
    	return down;
    }
 
    /**
     * Move the focus to the next component
     * Does NOT work! (cause nextFocus() doesn't do what I wanted)
     */
    public boolean keyDown( Event evt, int key ) {
    	if( key == (int)'\t' || key == Event.DOWN || key == Event.RIGHT )
	{
	    nextFocus();	// let's move to the next focus

	    // it seems not to work like this!
	    // It would work like this I think:
	    // getParent().getComponents()[me+1].requestFocus()
	    
	    return true;
	}
    	return false;
    }

    /**
     * This delivers the Event to the parent.
     */
    public void sendEvent() {
    	if( getParent() != null )
	    getParent().postEvent( new Event( this, Event.ACTION_EVENT, this ) );
    }
    
    /**
     * handle the MouseDownEvents.
     */
    public boolean mouseDown(Event event, int x, int y) {
    	if( isswitch == false )
	{
	    down = true;
	    current = pressed;
	    drawIt();
	}
	else {
	    if( down == true ) {
	    	down = false;
	    	current = normal;
	    }
	    else {
	        down = true;
	    	current = pressed;
	    }
	    drawIt();

	    // now send an Event, that the state has changed!
	    sendEvent();
	}

	return true;
    }

    /**
     * handle the MouseUpEvent.
     */
    public boolean mouseUp(Event event, int x, int y) {
	if( down && isswitch == false ) {
	    down = false;
	    current = normal;
	    drawIt();

	    // let's see if it's within my boundaries.
	    if((x >= 0) && (y >= 0) &&
	       (x < width) && (y < height))
		sendEvent();
	}
	if( isswitch == true ) { 
	    // think nothing has to be done here
	    ;
	}
	return true;
    }

    /**
     * updates the image on screen
     */
    void drawIt() {
	Graphics g = getGraphics();
	if( g != null ) paint( g );
    }
    
    /**
     * Draws the picture on the screen (if parts are loaded,
     * then it draws those parts).
     * Overrides paint-Method. 
     */
    public void paint(Graphics graphics) {
	boolean raise = true;
    
    	if( current != null && 
	    (checkImage( current, width, height, this ) & (WIDTH|HEIGHT|ALLBITS)) != 0 ) {
	    // we know width and height of it -> draw it.
	    // or the image is complete -> draw it
	    graphics.clearRect(0,0,width,height);
	    graphics.drawImage(current, 0, 0, width, height, this);
	}
	else {
	    if( text != null )
	        graphics.drawString( text, 2, height/2 );
	}

	if( nopush )
	   raise = true;    // always remain raised.
	else
	   raise = !down;

	// just show a border!
	graphics.setColor( Color.white );
	graphics.draw3DRect( 0, 0, width-1, height-1, raise );
	graphics.setColor( Color.white );
	graphics.draw3DRect( 1, 1, width-3, height-3, raise );

	/* some lostFocus() Events don't come!
	// and the nextFocus does not work!
	if( focus == true ) {
	    do something
	}
	*/
    }

    /**
     * tell my preferred-size
     */
    public Dimension preferredSize() {
	return new Dimension( width, height );
    }

    /**
     * tell my minimum-size
     */
    public Dimension minimumSize() {
	return new Dimension( width, height );
    }

    public boolean gotFocus( Event evt, Object what ) {
    	focus = true;
	drawIt();

	// pass it back
    	return false;
    }

    public boolean lostFocus( Event evt, Object what ) {
    	focus = false;
	drawIt();

	// pass it back
    	return false;
    }

    /**
     * Tell me all about the image ;-)
     * (inspired from Component.java)
     */
    public boolean imageUpdate(Image img, int flags,
                               int x, int y, int w, int h) {
        int rate = -1;

        if ((flags & (FRAMEBITS|ALLBITS)) != 0) {
            rate = 0;
        } else if ((flags & SOMEBITS) != 0) {
	    rate = 100;
        }
        if (rate >= 0) {
            repaint(rate, 0, 0, width, height);
        }
        return (flags & (ALLBITS|ABORT)) == 0;
    }
}

