import java.io.*;
import java.net.*;
import java.util.*;

/**
 * <B>FMUserDatabase class</B>.<BR><BR>
 * Implements all the functions to maintain a user database
 * and to do some simple queries on it.
 */
public class FMUserDatabase {
  private static final int FIRST_USER_ID = 1000;
  private String databaseName; 
	
  /**
   * Public constructor.
   * Initializes the database.
   * 
   * @param databaseName A valid directory name where the database will be stored
   */
  public FMUserDatabase(String databaseName) {
    this.databaseName = databaseName;
  };
	
  /**
   * Return a vector with all the registered IDs.
   */
  private Vector getUsers() {
    File databaseFile;
    String[] userFile;
    Vector toRet;
    int i;
    
    toRet = new Vector(10, 5); 
		
    databaseFile = new File(databaseName);
    userFile = databaseFile.list();
		
    for(i = 0; i < userFile.length; i++) {
      int ID;
			
      try {
	ID = Integer.parseInt(userFile[i]);
      }
      catch(NumberFormatException exception) {
	continue;
      };
			
      toRet.addElement(new Integer(ID));
    };

    return toRet;
  };

  /**
   * Get the first available user ID in the database.
   *
   * @return The first available user ID.
   */
  public int getNewID() {
    Vector users;
    int toRet;
		
    users = getUsers();
		
    toRet = FIRST_USER_ID;
    while(true) {
      Enumeration e;
      Integer ID;
      boolean found;
    
      found = false;
      e = users.elements();
      while(e.hasMoreElements()) {
	ID = (Integer)e.nextElement();
	if(toRet == ID.intValue()) {
	  found = true;
	  break;
	};
      };
			
      if(!found) break;
			
      toRet++;
    };
	
    return toRet;
  };
	
  /**
   * Search for a user matching the specified ID.
   *
   * @param ID The ID of the desired user.
   * @exception java.io.IOException No user matching the ID found.
   * @return The user with the specified ID.
   */
  public FMUser getUser(int ID) throws IOException {
    FMUser toRet;
    
    toRet = new FMUser();
    toRet.read(databaseName + "/" + ID);
    
    return toRet;
  };

  /**
   * Add a new user to the database.
   * 
   * @param user The user to be added.
   * @exception java.io.IOException Error writing the user file.
   */
  public void addUser(FMUser user) throws IOException {
    user.setID(getNewID());
    user.write(databaseName + "/" + user.getID());
  };

  /**
   * Update an existing user in the database.
   *
   * @param user The user to be updated.
   * @exception java.io.IOException Error writing the user file.
   */
  public void updateUser(FMUser user) throws IOException {
    user.write(databaseName + "/" + user.getID());
  };

  /**
   * Update the password of a specified user.
   *
   * @param ID The user ID.
   * @param password The new password.
   * @exception java.io.IOException Error writing the user file.
   */
  public void setUserPassword(int ID, String password) throws IOException {
    FMUser user;

    user = getUser(ID);
    user.setPassword(password);
    updateUser(user);
  };

  /**
   * Update the nickname of a specified user.
   *
   * @param ID The user ID.
   * @param password The new nickname.
   * @exception java.io.IOException Error writing the user file.
   */
  public void setUserNickName(int ID, String nickName) throws IOException {
    FMUser user;

    user = getUser(ID);
    user.setNickName(nickName);
    updateUser(user);
  };

  /**
   * Update the real name of a specified user.
   *
   * @param ID The user ID.
   * @param realName The new realName.
   * @exception java.io.IOException Error writing the user file.
   */
  public void setUseRealName(int ID, String realName) throws IOException {
    FMUser user;

    user = getUser(ID);
    user.setRealName(realName);
    updateUser(user);
  };

  /**
   * Update the EMail of a specified user.
   *
   * @param ID The user ID.
   * @param eMail The new e-mail.
   * @exception java.io.IOException Error writing the user file.
   */
  public void setUserEMail(int ID, String eMail) throws IOException {
    FMUser user;

    user = getUser(ID);
    user.setEMail(eMail);
    updateUser(user);
  };

  /**
   * Update the info of a specified user.
   *
   * @param ID The user ID.
   * @param info The new info.
   * @exception java.io.IOException Error writing the user file.
   */
  public void setUserInfo(int ID, String info) throws IOException {
    FMUser user;

    user = getUser(ID);
    user.setInfo(info);
    updateUser(user);
  };

  /**
   * Query the user database by user ID.
   *
   * @param ID The user ID.
   * @param return A vector with all the matchings.
   * @exception java.io.IOException Error reading the user file.
   */
  public Vector queryByID(int ID) throws IOException {
    Vector users;
    Vector matchings;
    Enumeration e;

    matchings = new Vector(10, 5);

    users = getUsers();
    e = users.elements();
    
    while(e.hasMoreElements()) {
      Integer currentUserID;
      FMUser currentUser;

      currentUserID = (Integer)e.nextElement();
      currentUser = getUser(currentUserID.intValue());
      
      if(ID == currentUser.getID()) matchings.addElement(currentUser);
    };

    return matchings;
  };

  /**
   * Query the user database by nick name.
   *
   * @param nickName The nick name.
   * @param return A vector with all the matchings.
   * @exception java.io.IOException Error reading the user file.
   */
  public Vector queryByNickName(String nickName) throws IOException {
    Vector users;
    Vector matchings;
    Enumeration e;

    matchings = new Vector(10, 5);

    users = getUsers();
    e = users.elements();
    
    while(e.hasMoreElements()) {
      Integer currentUserID;
      FMUser currentUser;

      currentUserID = (Integer)e.nextElement();
      currentUser = getUser(currentUserID.intValue());
      
      if(currentUser.getNickName().toUpperCase().lastIndexOf(nickName.toUpperCase()) != -1) matchings.addElement(currentUser);
    };

    return matchings;
  };

  /**
   * Query the user database by real name.
   *
   * @param realName The real name.
   * @param return A vector with all the matchings.
   * @exception java.io.IOException Error reading the user file.
   */
  public Vector queryByRealName(String realName) throws IOException {
    Vector users;
    Vector matchings;
    Enumeration e;

    matchings = new Vector(10, 5);

    users = getUsers();
    e = users.elements();
    
    while(e.hasMoreElements()) {
      Integer currentUserID;
      FMUser currentUser;

      currentUserID = (Integer)e.nextElement();
      currentUser = getUser(currentUserID.intValue());
      
      if(currentUser.getRealName().toUpperCase().lastIndexOf(realName.toUpperCase()) != -1) matchings.addElement(currentUser);
    };

    return matchings;
  };

  /**
   * Query the user database by e-mail.
   *
   * @param eMail The e-mail.
   * @param return A vector with all the matchings.
   * @exception java.io.IOException Error reading the user file.
   */
  public Vector queryByEMail(String eMail) throws IOException {
    Vector users;
    Vector matchings;
    Enumeration e;

    matchings = new Vector(10, 5);

    users = getUsers();
    e = users.elements();
    
    while(e.hasMoreElements()) {
      Integer currentUserID;
      FMUser currentUser;

      currentUserID = (Integer)e.nextElement();
      currentUser = getUser(currentUserID.intValue());
      
      if(currentUser.getEMail().toUpperCase().lastIndexOf(eMail.toUpperCase()) != -1) matchings.addElement(currentUser);
    };

    return matchings;
  };
};		
			


